package io.jenkins.plugins.portshift_vulnerability_scanner;

import hudson.AbortException;
import hudson.BulkChange;
import hudson.Extension;
import hudson.FilePath;
import hudson.Launcher;
import hudson.model.AbstractProject;
import hudson.model.Run;
import hudson.model.TaskListener;
import hudson.tasks.BuildStepDescriptor;
import hudson.tasks.Builder;
import hudson.util.FormValidation;
import hudson.util.Secret;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Base64;
import java.util.UUID;
import javax.annotation.Nonnull;
import jenkins.tasks.SimpleBuildStep;
import net.sf.json.JSONObject;
import org.apache.commons.lang.StringUtils;
import org.jenkinsci.Symbol;
import org.kohsuke.stapler.DataBoundConstructor;
import org.kohsuke.stapler.DataBoundSetter;
import org.kohsuke.stapler.QueryParameter;
import org.kohsuke.stapler.StaplerRequest;


public class PortshiftVulnerabilityScannerBuilder extends Builder implements SimpleBuildStep {

  protected static final String PLUGIN_NAME = "portshift-ci";
  private static final String IMAGE_NAME_REGEX = "^[a-z0-9]([-a-z0-9]*[a-z0-9])?$";
  private static final String IMAGE_TAG_REGEX = "\\S+"; //String of at least one character where all characters are non-whitespace

  private String accessKey;
  private Secret secretKey;
  private String imageName;
  private String imageTag;
  private String url;
  private String scannerIp;


  @DataBoundConstructor
  public PortshiftVulnerabilityScannerBuilder(String accessKey, String secretKey, String imageName, String imageTag,
      String url, String scannerIp) {
    this.accessKey = accessKey;
    this.secretKey = Secret.fromString(secretKey);
    this.imageName = imageName;
    this.imageTag = imageTag;
    this.scannerIp = scannerIp;
    if (url.contains("://")) {
      this.url = url.split("://")[1];
    } else {
      this.url = url;
    }
  }

  @Override
  public void perform(@Nonnull Run<?, ?> run, @Nonnull FilePath workspace, @Nonnull Launcher launcher,
      @Nonnull TaskListener listener)
      throws AbortException {

    if (accessKey == null || secretKey == null || imageName == null || imageTag == null || url == null
        || scannerIp == null) {
      throw new AbortException(
          "Missing configuration. Please set the plugin configuration parameters before continuing.\n");
    }

    int exitCode = PortshiftVulnerabilityScanExecuter
        .execute(run, workspace, launcher, listener, accessKey, secretKey, imageName, imageTag, url, scannerIp);
    listener.getLogger().println("exitCode: " + exitCode);

    String failedMessage = "Vulnerability scan has failed.";
    switch (exitCode) {
      case 0:
        listener.getLogger().println("Vulnerability scan has completed successfully.");
        break;
      case 4:
        throw new AbortException(failedMessage);
      default:
        // This exception causes the message to appear in the Jenkins console
        throw new AbortException(failedMessage);
    }
  }

  // Overridden for better type safety.
  // If your plugin doesn't really define any property on Descriptor,
  // you don't have to do this.
  @Override
  public DescriptorImpl getDescriptor() {
    return (DescriptorImpl) super.getDescriptor();
  }


  /**
   * Descriptor for {@link io.jenkins.plugins.portshift_vulnerability_scanner.PortshiftVulnerabilityScannerBuilder}.
   * Used as a singleton. The class is marked as public so that it can be accessed from views.
   */
  @Symbol("portshift_vulnerability_scanner")
  @Extension // This indicates to Jenkins that this is an implementation of an extension
  // point.
  public static final class DescriptorImpl extends BuildStepDescriptor<Builder> {

    private static final String IP_PATTERN = "^((0|1\\d?\\d?|2[0-4]?\\d?|25[0-5]?|[3-9]\\d?)\\.){3}(0|1\\d?\\d?|2[0-4]?\\d?|25[0-5]?|[3-9]\\d?)$";
    public static final String LATEST_TAG = "latest";


    /**
     * To persist global configuration information, simply store it in a field and call save().
     */
    @SuppressWarnings("unused")
    private String accessKey;

    @SuppressWarnings("unused")
    private Secret secretKey;

    @SuppressWarnings("unused")
    private String imageName;

    @SuppressWarnings("unused")
    private String imageTag;

    @SuppressWarnings("unused")
    private String url;

    @SuppressWarnings("unused")
    private String scannerIp;

    /**
     * In order to load the persisted global configuration, you have to call load() in the constructor.
     */
    public DescriptorImpl() {
      load();
    }


    @SuppressWarnings("ResultOfMethodCallIgnored")
    public FormValidation doCheckAccessKey(@QueryParameter final String accessKey) {
      try {
        UUID.fromString(accessKey);
        return FormValidation.ok();
      } catch (Exception e) {
        return FormValidation
            .error(Messages.PortshiftVulnerabilityScannerBuilder_DescriptorImpl_errors_invalidAccesskey());
      }
    }

    public FormValidation doCheckSecretKey(@QueryParameter final String secretKey) {
      try {
        Base64.Decoder decoder = Base64.getDecoder();
        decoder.decode(secretKey);
        return FormValidation.ok();
      } catch (Exception e) {
        return FormValidation
            .error(Messages.PortshiftVulnerabilityScannerBuilder_DescriptorImpl_errors_invalidSecretkey());
      }
    }


    public FormValidation doCheckImageName(@QueryParameter final String imageName) {
      try {
        if (!imageName.matches(IMAGE_NAME_REGEX)) {
          return FormValidation
              .error(Messages.PortshiftVulnerabilityScannerBuilder_DescriptorImpl_errors_invalidImageName());
        }
        return FormValidation.ok();
      } catch (Exception e) {
        return FormValidation
            .error(Messages.PortshiftVulnerabilityScannerBuilder_DescriptorImpl_errors_invalidImageName());
      }
    }

    public FormValidation doCheckImageTag(@QueryParameter final String imageTag) {
      try {
        if (!StringUtils.isEmpty(imageTag) && !imageTag.matches(IMAGE_TAG_REGEX)) {
          return FormValidation
              .error(Messages.PortshiftVulnerabilityScannerBuilder_DescriptorImpl_errors_invalidImageTag());
        }
        return FormValidation.ok();
      } catch (Exception e) {
        return FormValidation
            .error(Messages.PortshiftVulnerabilityScannerBuilder_DescriptorImpl_errors_invalidImageTag());
      }
    }

    public FormValidation doCheckUrl(@QueryParameter final String url) {
      try {
        if (!isValidUrl(url) && !isValidUrl("https://" + url)) {
          return FormValidation.error(Messages.PortshiftVulnerabilityScannerBuilder_DescriptorImpl_errors_invalidUrl());
        }
        return FormValidation.ok();
      } catch (Exception e) {
        return FormValidation.error(Messages.PortshiftVulnerabilityScannerBuilder_DescriptorImpl_errors_invalidUrl());
      }
    }

    public FormValidation doCheckScannerIp(@QueryParameter final String scannerIp) {
      try {
        if (!isValidIp(scannerIp)) {
          return FormValidation
              .error(Messages.PortshiftVulnerabilityScannerBuilder_DescriptorImpl_errors_invalidScannerIp());
        }
        return FormValidation.ok();
      } catch (Exception e) {
        return FormValidation
            .error(Messages.PortshiftVulnerabilityScannerBuilder_DescriptorImpl_errors_invalidScannerIp());
      }
    }

    private boolean isValidUrl(String url) {
      try {
        new URL(url);
        return true;
      } catch (MalformedURLException e) {
        return false;
      }
    }

    public static boolean isValidIp(final String ip) {
      return ip.matches(IP_PATTERN);
    }


    public boolean isApplicable(Class<? extends AbstractProject> aClass) {
      // Indicates that this builder can be used with all kinds of project types
      return true;
    }

    /**
     * This human readable name is used in the configuration screen.
     */
    @Override
    @Nonnull
    public String getDisplayName() {
      return Messages.PortshiftVulnerabilityScannerBuilder_DescriptorImpl_DisplayName();
    }

    @Override
    public boolean configure(StaplerRequest req, JSONObject formData) throws FormException {

      try (BulkChange bc = new BulkChange(this)) {
        req.bindJSON(this, formData);
        bc.commit();
      } catch (IOException e) {
        throw new FormException(e,"failed to parse form data");
      }
      return true;

    }

    @DataBoundSetter
    @SuppressWarnings("unused")
    public DescriptorImpl setAccessKey(@SuppressWarnings("unused") String accessKey) {
      this.accessKey = accessKey;
      save();
      return this;
    }

    @DataBoundSetter
    public DescriptorImpl setSecretKey(Secret secretKey) {
      this.secretKey = secretKey;
      save();
      return this;
    }

    @DataBoundSetter
    public DescriptorImpl setImageName(String imageName) {
      this.imageName = imageName;
      save();
      return this;
    }

    @DataBoundSetter
    public DescriptorImpl setUrl(String url) {
      this.url = url;
      save();
      return this;
    }

    @DataBoundSetter
    public DescriptorImpl setScannerIp(String scannerIp) {
      this.scannerIp = scannerIp;
      save();
      return this;
    }

    @DataBoundSetter
    public DescriptorImpl setImageTag(String imageTag) {
      if (StringUtils.isEmpty(imageTag)) {
        imageTag = LATEST_TAG;
      }
      this.imageTag = imageTag;
      return this;
    }
  }
}
