package org.aerogear.kryptowire.workflow;

import hudson.Extension;
import hudson.FilePath;
import hudson.model.Run;
import hudson.model.TaskListener;
import jenkins.model.GlobalConfiguration;
import org.aerogear.kryptowire.BinaryInfo;
import org.aerogear.kryptowire.GlobalConfigurationImpl;
import org.aerogear.kryptowire.KryptowireService;
import org.aerogear.kryptowire.KryptowireServiceImpl;
import org.aerogear.kryptowire.BinaryHistoryAction;
import org.apache.commons.lang.StringUtils;
import org.jenkinsci.plugins.workflow.steps.AbstractStepDescriptorImpl;
import org.jenkinsci.plugins.workflow.steps.AbstractStepImpl;
import org.jenkinsci.plugins.workflow.steps.AbstractSynchronousNonBlockingStepExecution;
import org.jenkinsci.plugins.workflow.steps.StepContextParameter;
import org.json.JSONObject;
import org.kohsuke.stapler.DataBoundConstructor;
import org.kohsuke.stapler.DataBoundSetter;

import javax.annotation.Nonnull;
import javax.inject.Inject;

public class KWSubmitStep extends AbstractStepImpl {

    private String platform;
    private String filePath;
    private String kwAPIKey;
    private String appiumScript;

    public String getPlatform() {
        return platform;
    }

    @DataBoundSetter
    public void setPlatform(String platform) {
        this.platform = platform;
    }

    public String getFilePath() {
        return filePath;
    }

    @DataBoundSetter
    public void setFilePath( String filePath ){
        this.filePath = filePath;
    }

    public String getApiKey() {
        return kwAPIKey;
    }

    @DataBoundSetter
    public void setApiKey(String apiKey) {
        this.kwAPIKey = apiKey;
    }

    public String getAppiumScript() {
        return appiumScript;
    }

    @DataBoundSetter
    public void setAppiumScript(String appiumScript) {
        this.appiumScript = appiumScript;
    }

    @DataBoundConstructor
    public KWSubmitStep(@Nonnull String platform, @Nonnull String filePath) {
        this.platform = platform;
        this.filePath = filePath;
    }

    @Extension
    public static class DescriptorImpl extends AbstractStepDescriptorImpl {

        public DescriptorImpl() {
            super(KWSubmitExecution.class);
        }

        @Override
        public String getFunctionName() {
            return "kwSubmit";
        }

        @Override
        public String getDisplayName() {
            return "Submit to Kryptowire";
        }

    }

    public static class KWSubmitExecution extends AbstractSynchronousNonBlockingStepExecution<Void> {

        private static final long serialVersionUID = 1L;

        @Inject
        transient KWSubmitStep step;

        @StepContextParameter
        transient TaskListener listener;

        @StepContextParameter
        private transient Run build;

        @Override
        protected Void run() throws Exception {

            GlobalConfigurationImpl pluginConfig = GlobalConfiguration.all().get(GlobalConfigurationImpl.class);

            if (pluginConfig == null) {
                throw new RuntimeException("[Error] Could not retrieve global Kryptowire config object.");
            }


            String kwEndpoint = pluginConfig.getKwEndpoint();
            String kwApiKey = step.kwAPIKey;
            listener.getLogger().println(" --- APIKey: " + kwApiKey);

            // LEAVE THIS FOR LEGACY SUPPORT: can specify API Key in the config, but this is insecure! 
            if ( kwApiKey == null || kwApiKey.isEmpty() ){
                kwApiKey = pluginConfig.getKwApiKey();
            }
            if( StringUtils.isEmpty(kwEndpoint) ){
                throw new RuntimeException("Kryptowire plugin configuration is not correct: Endpoint is not set!");
            }
            if ( StringUtils.isEmpty(kwApiKey)) {
                throw new RuntimeException("Kryptowire plugin configuration is not correct: API Key is not passed in the Jenkinsfile, nor set in the plugin configuration.");
            }

            FilePath fp = getContext().get(FilePath.class).child(step.filePath);

            FilePath appiumScriptFp = null;
            if (step.appiumScript != null && !step.appiumScript.isEmpty()) {
                appiumScriptFp = getContext().get(FilePath.class).child(step.appiumScript);
            }

            listener.getLogger().println(" --- Kryptowire submit Start ---");
            listener.getLogger().println("kwSubmit: " + step.platform + " : " + step.filePath);
            if (appiumScriptFp != null) {
                listener.getLogger().println("kwSubmit appium_script: " + step.appiumScript);
            }

            KryptowireService kws = new KryptowireServiceImpl(kwEndpoint,  kwApiKey);

            JSONObject resp = kws.submit(step.platform, fp, appiumScriptFp);

            String uuid = resp.getString("uuid");
            String platform = resp.getString("platform");
            String pkg = resp.getString("package");
            String version = resp.getString("version");
            String hash = resp.getString("hash");

            listener.getLogger().println("kw msg: " + resp.get("msg"));
            listener.getLogger().println("kw uuid: " + uuid);
            listener.getLogger().println("kw platform: " + platform);
            listener.getLogger().println("kw package: " + pkg);
            listener.getLogger().println("kw version: " + version);
            listener.getLogger().println("kw hash: " + hash);

            BinaryInfo info = BinaryInfo.fromJSONObject(resp);
            build.addAction(new BinaryHistoryAction(info));

            listener.getLogger().println(" --- Kryptowire submit Done ---");

            return null;
        }

    }
}
