/*
 * Copyright (C) 2018 Alauda.io
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.alauda.jenkins.devops.sync.util;

import antlr.ANTLRException;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import hudson.model.*;
import hudson.model.queue.QueueTaskFuture;
import hudson.plugins.git.RevisionParameterAction;
import hudson.security.ACL;
import hudson.triggers.SCMTrigger;
import hudson.triggers.SafeTimerTask;
import hudson.triggers.TimerTrigger;
import hudson.triggers.Trigger;
import io.alauda.devops.java.client.models.*;
import io.alauda.jenkins.devops.sync.*;
import io.alauda.jenkins.devops.sync.action.AlaudaQueueAction;
import io.alauda.jenkins.devops.sync.client.Clients;
import io.kubernetes.client.models.V1ObjectMeta;
import jenkins.branch.BranchProjectFactory;
import jenkins.branch.MultiBranchProject;
import jenkins.model.Jenkins;
import jenkins.security.NotReallyRoleSensitiveCallable;
import jenkins.util.Timer;
import org.apache.commons.lang.StringUtils;
import org.eclipse.jgit.transport.URIish;
import org.jenkinsci.plugins.pipeline.modeldefinition.ast.ModelASTPipelineDef;
import org.jenkinsci.plugins.pipeline.modeldefinition.parser.Converter;
import org.jenkinsci.plugins.workflow.job.WorkflowJob;
import org.jenkinsci.plugins.workflow.job.WorkflowRun;
import org.jenkinsci.plugins.workflow.job.properties.PipelineTriggersJobProperty;
import org.jenkinsci.plugins.workflow.multibranch.WorkflowMultiBranchProject;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.validation.constraints.NotNull;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import static io.alauda.jenkins.devops.sync.constants.Constants.*;
import static io.alauda.jenkins.devops.sync.constants.PipelinePhases.*;
import static io.alauda.jenkins.devops.sync.util.AlaudaUtils.updatePipelinePhase;
import static java.util.logging.Level.SEVERE;

/**
 * @author suren
 */
public abstract class JenkinsUtils {
	private static final Logger LOGGER = Logger.getLogger(JenkinsUtils.class.getName());
	public static final String PARAM_FROM_ENV_DESCRIPTION = "From Alauda DevOps PipelineConfig Parameter";

	private JenkinsUtils(){}

	public static Job getJob(String job) {
		TopLevelItem item = Jenkins.getInstance().getItem(job);
		if (item instanceof Job) {
			return (Job) item;
		}
		return null;
	}

	@NotNull
	public static String getRootUrl() {
		// TODO is there a better place to find this?
		String root = Jenkins.getInstance().getRootUrl();
		return StringUtils.isBlank(root) ? ROOT_URL : root;
	}

	public static boolean verifyEnvVars(Map<String, ParameterDefinition> paramMap, WorkflowJob workflowJob) {
        if (paramMap != null) {
            String fullName = workflowJob.getFullName();
            WorkflowJob job = Jenkins.getInstance().getItemByFullName(fullName, WorkflowJob.class);
            if (job == null) {
                // this should not occur if an impersonate call has been made higher up
                // the stack
                LOGGER.warning(() -> "A run of workflow job " + workflowJob.getName() + " unexpectantly not saved to disk.");
                return false;
            }
            ParametersDefinitionProperty props = job.getProperty(ParametersDefinitionProperty.class);
            List<String> names = props.getParameterDefinitionNames();
            for (String name : names) {
                if (!paramMap.containsKey(name)) {
                    LOGGER.warning(() -> "A run of workflow job " + job.getName() + " was expecting parameter "
                            + name + ", but it is not in the parameter list");
                    return false;
                }
            }
        }
	    return true;
	}

    public static Map<String, ParameterDefinition> addJobParamForPipelineParameters(WorkflowJob job,
                                                                                    List<V1alpha1PipelineParameter> params, boolean replaceExisting) throws IOException {
        // get existing property defs, including any manually added from the
        // jenkins console independent of PC
        ParametersDefinitionProperty jenkinsParams = job.removeProperty(ParametersDefinitionProperty.class);

        Map<String, ParameterDefinition> paramMap = null;
        if (params != null && params.size() > 0) {
            // build list of current env var names for possible deletion of env
            // vars currently stored
            // as job params
            // builds a list of job parameters

            List<String> envKeys = new ArrayList<>();
            for (V1alpha1PipelineParameter parameter : params) {
                envKeys.add(parameter.getName());
            }
            paramMap = new HashMap<>();
            // store any existing parameters in map for easy key lookup
            if (jenkinsParams != null) {
                List<ParameterDefinition> existingParamList = jenkinsParams.getParameterDefinitions();
                for (ParameterDefinition param : existingParamList) {
                    // if a user supplied param, add
                    if (param.getDescription() == null || !param.getDescription().equals(PARAM_FROM_ENV_DESCRIPTION))
                        paramMap.put(param.getName(), param);
                    else if (envKeys.contains(param.getName())) {
                        // the env var still exists on the PipelineConfig side so
                        // keep
                        paramMap.put(param.getName(), param);
                    }
                }
            }

            for (V1alpha1PipelineParameter param : params) {
                ParameterDefinition jenkinsParam = null;
                switch (param.getType()) {
                    case PIPELINE_PARAMETER_TYPE_STRING_DEF:
                    case PIPELINE_PARAMETER_TYPE_STRING:
                        jenkinsParam = new StringParameterDefinition(param.getName(), param.getValue(), param.getDescription());
                        break;
                    case PIPELINE_PARAMETER_TYPE_BOOLEAN_DEF:
                    case PIPELINE_PARAMETER_TYPE_BOOLEAN:
                        jenkinsParam = new BooleanParameterDefinition(param.getName(), Boolean.valueOf(param.getValue()), param.getDescription());
                        break;
                    default:
                        LOGGER.warning(() -> "Parameter type `" + param.getType() + "` is not supported.. skipping...");
                        break;
                }

                if (jenkinsParam == null) {
                    continue;
                }
                // TODO: This is made differently from the original source
                // Need revisit this part if the parameters
                if (replaceExisting || !paramMap.containsKey(jenkinsParam.getName())) {
                    paramMap.put(jenkinsParam.getName(), jenkinsParam);
                }
            }

            List<ParameterDefinition> newParamList = new ArrayList<>(paramMap.values());
            job.addProperty(new ParametersDefinitionProperty(newParamList));
        }

        // force save here ... seen some timing issues with concurrent job updates and run initiations
        job.save();
        return paramMap;
    }

    /**
     * Override job's triggers
     * @param job
     * @param triggers
     * @return
     * @throws IOException
     */
    @NotNull
    public static List<ANTLRException> setJobTriggers(@Nonnull WorkflowJob job, List<V1alpha1PipelineTrigger> triggers) throws IOException {
        List<ANTLRException> exceptions = new ArrayList<>();
        if (CollectionUtils.isEmpty(triggers)) {
            return exceptions;
        }

        job.removeProperty(PipelineTriggersJobProperty.class);
        LOGGER.info(() -> "PipelineTrigger's count is " + triggers.size());

        for (V1alpha1PipelineTrigger pipelineTrigger : triggers) {
            Trigger trigger = null;
            final String type = pipelineTrigger.getType();
            if(type == null) {
                continue;
            }

            switch (type) {
                case PIPELINE_TRIGGER_TYPE_CODE_CHANGE:
                    V1alpha1PipelineTriggerCodeChange codeTrigger = pipelineTrigger.getCodeChange();

                    if (codeTrigger == null || !codeTrigger.isEnabled()) {
                        LOGGER.warning(() -> "Trigger type `" + PIPELINE_TRIGGER_TYPE_CODE_CHANGE + "` has empty description or is disabled...");
                        break;
                    }

                    try {
                        trigger = new SCMTrigger(codeTrigger.getPeriodicCheck());

                        LOGGER.info(() -> "Add CodeChangeTrigger.");
                    } catch (ANTLRException exc) {
                        LOGGER.log(Level.SEVERE, String.format("Error processing trigger type %s", PIPELINE_TRIGGER_TYPE_CODE_CHANGE), exc);
                        exceptions.add(exc);
                    }

                    break;
                case PIPELINE_TRIGGER_TYPE_CRON:
                    V1alpha1PipelineTriggerCron cronTrigger = pipelineTrigger.getCron();
                    if (cronTrigger == null || !cronTrigger.isEnabled()) {
                        LOGGER.warning(() -> "Trigger type `" + PIPELINE_TRIGGER_TYPE_CRON + "` has empty description or is disabled...");
                        break;
                    }

                    try {
                        trigger = new TimerTrigger(cronTrigger.getRule());

                        LOGGER.info(() -> "Add CronTrigger.");
                    } catch (ANTLRException exc) {
                        LOGGER.log(Level.SEVERE, String.format("Error processing trigger type %s", PIPELINE_TRIGGER_TYPE_CRON), exc);
                        exceptions.add(exc);
                    }

                    break;
                default:
                    LOGGER.warning(() -> "Trigger type `" + pipelineTrigger.getType() + "` is not supported... skipping...");
            }

            if(trigger != null) {
                job.addTrigger(trigger);
            }
        }

        LOGGER.info(() -> "Job trigger save done.");

        return exceptions;
    }

    @CheckForNull
	private static List<Action> putJobRunParamsFromEnvAndUIParams(List<V1alpha1PipelineParameter> pipelineParameters,
                                                                 List<Action> buildActions) {
        if(buildActions == null || pipelineParameters == null) {
            return buildActions;
        }

        List<ParameterValue> envVarList = getParameterValues(pipelineParameters);
        if (envVarList.size() == 0) {
            return buildActions;
        }

        buildActions.add(new ParametersAction(envVarList));

		return buildActions;
	}

	@Nonnull
    private static List<ParameterValue> getParameterValues(List<V1alpha1PipelineParameter> pipelineParameters) {
        List<ParameterValue> envVarList = new ArrayList<>();
        if (pipelineParameters == null) {
            return envVarList;
        }

        for (V1alpha1PipelineParameter pipeParam : pipelineParameters) {
            ParameterValue paramValue = null;
            String type = pipeParam.getType();
            if(type == null) {
                continue;
            }

            switch (type) {
                case PIPELINE_PARAMETER_TYPE_STRING_DEF:
                case PIPELINE_PARAMETER_TYPE_STRING:
                    paramValue = new StringParameterValue(pipeParam.getName(),
                            pipeParam.getValue(), pipeParam.getDescription());
                    break;
                case PIPELINE_PARAMETER_TYPE_BOOLEAN_DEF:
                case PIPELINE_PARAMETER_TYPE_BOOLEAN:
                    paramValue = new BooleanParameterValue(pipeParam.getName(),
                            Boolean.valueOf(pipeParam.getValue()), pipeParam.getDescription());
                    break;
                default:
                    LOGGER.warning(() -> "Parameter type `" + pipeParam.getType() + "` is not supported.. skipping...");
                    break;
            }

            if (paramValue != null) {
                envVarList.add(paramValue);
            }
        }

        return envVarList;
    }

    public static boolean triggerJob(@Nonnull WorkflowJob job, @Nonnull V1alpha1Pipeline pipeline)
            throws IOException {
        final V1ObjectMeta pipMeta = pipeline.getMetadata();
        final String namespace = pipMeta.getNamespace();
        final String pipelineName = pipMeta.getName();
	    LOGGER.info(() -> "will trigger pipeline: " + pipelineName);

        if (isAlreadyTriggered(job, pipeline)) {
            LOGGER.info(() -> "pipeline already triggered: "+pipelineName);
            return false;
        }

        AlaudaJobProperty pcProp = job.getProperty(WorkflowJobProperty.class);
        if (pcProp == null) {
            if(job.getParent() instanceof WorkflowMultiBranchProject) {
                pcProp = ((WorkflowMultiBranchProject) job.getParent()).getProperties().get(MultiBranchProperty.class);
            }
        }

        if(pcProp == null) {
            LOGGER.warning(() -> "aborting trigger of pipeline " + pipeline.getMetadata().getNamespace() + "/"+ pipeline.getMetadata().getName()
                    + "because of missing pc project property");
            return false;
        }

        final String pipelineConfigName = pipeline.getSpec().getPipelineConfig().getName();
        V1alpha1PipelineConfig pipelineConfig = Clients.get(V1alpha1PipelineConfig.class).lister().namespace(namespace).get(pipelineConfigName);
        if (pipelineConfig == null) {
            LOGGER.info(() -> "pipeline config not found....: "+pipelineName+" - config name "+pipelineConfigName);
            return false;
        }

        // sync on intern of name should guarantee sync on same actual obj
        synchronized (pipelineConfig.getMetadata().getUid().intern()) {
          LOGGER.info(() -> "pipeline config source credentials: "+pipelineConfig.getMetadata().getName());

            // We need to ensure that we do not remove
            // existing Causes from a Run since other
            // plugins may rely on them.
            List<Cause> newCauses = new ArrayList<>();
            newCauses.add(new JenkinsPipelineCause(pipeline, pcProp.getUid()));
            CauseAction originalCauseAction = PipelineToActionMapper.removeCauseAction(pipelineName);
            if (originalCauseAction != null) {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Adding existing causes...");
                    for (Cause c : originalCauseAction.getCauses()) {
                        LOGGER.log(Level.FINE, "trigger error", c);
                    }
                }
                newCauses.addAll(originalCauseAction.getCauses());
                if (LOGGER.isLoggable(Level.FINE)) {
                    for (Cause c : newCauses) {
                        LOGGER.log(Level.FINE, "trigger error", c);
                    }
                }
            }

            List<Action> pipelineActions = new ArrayList<>();
            CauseAction bCauseAction = new CauseAction(newCauses);
            pipelineActions.add(bCauseAction);
            pipelineActions.add(new AlaudaQueueAction());

            V1alpha1PipelineSourceGit sourceGit = pipeline.getSpec().getSource().getGit();
            String commit = null;
            if (pipMeta.getAnnotations() != null && pipMeta.getAnnotations().containsKey(ALAUDA_DEVOPS_ANNOTATIONS_COMMIT)) {
              commit = pipMeta.getAnnotations().get(ALAUDA_DEVOPS_ANNOTATIONS_COMMIT);
            }

          if (sourceGit != null && commit != null) {
            try {
              URIish repoURL = new URIish(sourceGit.getUri());
              pipelineActions.add(new RevisionParameterAction(commit, repoURL));
            } catch (URISyntaxException e) {
              LOGGER.log(SEVERE, "Failed to parse git repo URL" + sourceGit.getUri(), e);
            }
          }

          LOGGER.info("pipeline got cause....: "+pipelineName+" pipeline actions "+pipelineActions);

            // params added by user in jenkins ui
            PipelineToActionMapper.removeParameterAction(pipelineName);

            putJobRunParamsFromEnvAndUIParams(pipeline.getSpec().getParameters(), pipelineActions);

            LOGGER.info(() -> "pipeline config update with job: "+pipelineName+" pipeline config "+pipelineConfig.getMetadata().getName());

            Action[] actionArray;
            if(pipelineActions.size() == 0) {
                actionArray = new Action[]{};
            } else {
                actionArray = pipelineActions.toArray(new Action[0]);
            }

            QueueTaskFuture<WorkflowRun> queueTaskFuture = job.scheduleBuild2(0, actionArray);
            if (queueTaskFuture != null) {
                // TODO should offer a better solution
                // TODO should we add an extension point here?
                if(job.getParent() instanceof MultiBranchProject) {
                    BranchProjectFactory factory = ((MultiBranchProject) job.getParent()).getProjectFactory();

                    SCMRevisionAction revisionAction = null;
                    for(Action action : actionArray) {
                        if(action instanceof CauseAction) {
                            List<Cause> causes = ((CauseAction) action).getCauses();
                            if(causes != null) {
                                for(Cause cause : causes) {
                                    if(cause instanceof SCMRevisionAction) {
                                        revisionAction = (SCMRevisionAction) cause;
                                        break;
                                    }
                                }
                            }
                        }
                    }

                    if(revisionAction != null) {
                        factory.setRevisionHash(job, revisionAction.getRevision());
                    }
                }

                pipeline.getStatus().setPhase(QUEUED);

                // If builds are queued too quickly, Jenkins can add the cause
                // to the previous queued pipeline so let's add a tiny
                // sleep.
                try {
                    TimeUnit.MILLISECONDS.sleep(50);
                } catch (InterruptedException e) {
                    LOGGER.log(Level.SEVERE, "updatePipelinePhase Interrupted", e);
                    Thread.currentThread().interrupt();
                }
                return true;
            }
            pipeline.getStatus().setPhase(FAILED);
            LOGGER.info(() -> "Will not schedule build for this pipeline: "+pipelineName);

            return false;
        }
    }

	private static boolean isAlreadyTriggered(WorkflowJob job, V1alpha1Pipeline pipeline) {
		return getRun(job, pipeline) != null;
	}

	public synchronized static void cancelPipeline(WorkflowJob job, V1alpha1Pipeline pipeline) {
		cancelPipeline(job, pipeline, false);
	}

	public synchronized static void cancelPipeline(WorkflowJob job, V1alpha1Pipeline pipeline, boolean deleted) {
		if (!cancelQueuedPipeline(job, pipeline)) {
            cancelRunningPipeline(job, pipeline);
		}

		if (deleted) {
			return;
		}

        updatePipelinePhase(pipeline, CANCELLED);
	}

	private static WorkflowRun getRun(WorkflowJob job, V1alpha1Pipeline pipeline) {
		if (pipeline != null && pipeline.getMetadata() != null) {
			return getRun(job, pipeline.getMetadata().getUid());
		}
		return null;
	}

    private static WorkflowRun getRun(WorkflowJob job, String pipelineUid) {
        for (WorkflowRun run : job.getBuilds()) {
            JenkinsPipelineCause cause = PipelineUtils.findAlaudaCause(run);
            if (cause != null && cause.getUid().equals(pipelineUid)) {
                return run;
            }
        }
        return null;
	}

	public synchronized static void deleteRun(WorkflowRun run) {
        try {
            LOGGER.info("Deleting run: " + run.toString());
            run.delete();
        } catch (IOException e) {
            LOGGER.warning(() -> "Unable to delete run " + run.toString() + ":" + e.getMessage());
        }
	}

	private static boolean cancelRunningPipeline(WorkflowJob job, V1alpha1Pipeline pipeline) {
		String pipelineUid = pipeline.getMetadata().getUid();
		WorkflowRun run = getRun(job, pipelineUid);
		if (run != null && run.isBuilding()) {
			terminateRun(run);
			return true;
		}
		return false;
	}

	private static boolean cancelNotYetStartedPipeline(WorkflowJob job, V1alpha1Pipeline pipeline) {
		String pipelineUid = pipeline.getMetadata().getUid();
		WorkflowRun run = getRun(job, pipelineUid);
		if (run != null && run.hasntStartedYet()) {
			terminateRun(run);
			return true;
		}
		return false;
	}

	public static void terminateRun(final WorkflowRun run) {
		ACL.impersonate(ACL.SYSTEM, new NotReallyRoleSensitiveCallable<Void, RuntimeException>() {
			@Override
			public Void call() throws RuntimeException {
				run.doTerm();
				Timer.get().schedule(new SafeTimerTask() {
					@Override
					public void doRun() {
						ACL.impersonate(ACL.SYSTEM, new NotReallyRoleSensitiveCallable<Void, RuntimeException>() {
							@Override
							public Void call() throws RuntimeException {
								run.doKill();
								return null;
							}
						});
					}
				}, 5, TimeUnit.SECONDS);
				return null;
			}
		});
	}

	@SuppressFBWarnings("SE_BAD_FIELD")
	private static boolean cancelQueuedPipeline(WorkflowJob job, V1alpha1Pipeline pipeline) {
	    LOGGER.info("cancelling queued pipeline: "+pipeline.getMetadata().getName());
        String pipelineUid = pipeline.getMetadata().getUid();
        final Queue pipelineQueue = Jenkins.getInstance().getQueue();
        boolean foundInQueue = false;
        for (final Queue.Item item : pipelineQueue.getItems()) {
            for (JenkinsPipelineCause cause : PipelineUtils.findAllAlaudaCauses(item)) {
                if (cause.getUid().equals(pipelineUid)) {
                    foundInQueue = true;
                    return ACL.impersonate(ACL.SYSTEM, new NotReallyRoleSensitiveCallable<Boolean, RuntimeException>() {
                        @Override
                        public Boolean call() throws RuntimeException {
                            pipelineQueue.cancel(item);
                            return true;
                        }
                    });
                }
            }
        }

        if(!foundInQueue) {
            LOGGER.info("Not found pipeline: %s" + pipeline.getMetadata().getName());
        }

		return cancelNotYetStartedPipeline(job, pipeline);
	}

	public static void cancelQueuedBuilds(WorkflowJob job, String pcUid) {
        LOGGER.info(() -> "cancelling queued pipeline by uuid: "+pcUid);
		Queue pipelineQueue = Jenkins.getInstance().getQueue();
		for (Queue.Item item : pipelineQueue.getItems()) {
            JenkinsPipelineCause pipelineCause = PipelineUtils.findAlaudaCause(item);
            if(pipelineCause == null) {
                continue;
            }

            if (pipelineCause.getPipelineConfigUid().equals(pcUid)) {
                V1alpha1Pipeline pipeline = new V1alpha1PipelineBuilder().withMetadata(new V1ObjectMeta().namespace(pipelineCause.getNamespace())
                        .name(pipelineCause.getName())).build();
                cancelQueuedPipeline(job, pipeline);
            }
		}
	}



//    public static void maybeScheduleNext(WorkflowJob job) {
//        WorkflowJobProperty pcp = WorkflowJobUtils.getAlaudaProperty(job);
//        if (pcp == null) {
//            return;
//        }
//
//        List<V1alpha1Pipeline> pipelines = PipelineResourceSyncController.getCurrentPipelineController()
//                .listPipelines(pcp.getNamespace())
//                .stream()
//                .filter(pipe -> {
//                    Map<String, String> labels = pipe.getMetadata().getLabels();
//                    if (labels == null) {
//                        return false;
//                    }
//                    return pcp.getName().equals(labels.get(ALAUDA_DEVOPS_LABELS_PIPELINE_CONFIG)) && pipe.getStatus().getPhase().equals(PENDING);
//                }).collect(Collectors.toList());
//
//        handlePipelineList(job, pipelines);
//    }
//
//	public static void handlePipelineList(WorkflowJob job, List<V1alpha1Pipeline> pipelines) {
//		if (pipelines.isEmpty()) {
//			return;
//		}
//        pipelines.sort(new PipelineComparator());
//		boolean isSerial = !job.isConcurrentBuild();
//		boolean jobIsBuilding = job.isBuilding();
//		for (int i = 0; i < pipelines.size(); i++) {
//            V1alpha1Pipeline p = pipelines.get(i);
//			if (!AlaudaUtils.isPipelineStrategyPipeline(p))
//				continue;
//			// For SerialLatestOnly we should try to cancel all pipelines before
//			// the latest one requested.
//            if (jobIsBuilding && !isCancelled(p.getStatus())) {
//                return;
//            }
//
//            if (i < pipelines.size() - 1) {
//                cancelQueuedPipeline(job, p);
//                // TODO: maybe not necessary?
//                updatePipelinePhase(p, CANCELLED);
//                continue;
//            }
//
//			boolean buildAdded = false;
//			try {
//				buildAdded = PipelineResourceSyncController.addEventToJenkinsJobRun(p);
//			} catch (IOException e) {
//				V1ObjectMeta meta = p.getMetadata();
//				LOGGER.log(WARNING, "Failed to add new build " + meta.getNamespace() + "/" + meta.getName(), e);
//			}
//			// If it's a serial build then we only need to schedule the first
//			// build request.
//			if (isSerial && buildAdded) {
//				return;
//			}
//		}
//	}

    @Nonnull
    public static String getFullJobName(@Nonnull WorkflowJob job) {
		return job.getRelativeNameFrom(Jenkins.getInstance());
	}

	@Nonnull
    public static String formatJenkinsfile(String unformattedJenkinsfile) throws IOException {
        ModelASTPipelineDef pipelineDef = Converter.scriptToPipelineDef(unformattedJenkinsfile);
        if (pipelineDef == null) {
            throw new IOException("Jenkinsfile content '" + unformattedJenkinsfile + "' did not contain the 'pipeline' step or miss some steps");
        }
        return pipelineDef.toPrettyGroovy();
    }

    /**
     * TODO consider gather with other methods
     * @param run
     * @return
     */
	public static boolean fromMultiBranch(@NotNull Run run) {
        Job wfJob = run.getParent();
        if(!(wfJob instanceof WorkflowJob)) {
            return false;
        }

        return (wfJob.getParent() instanceof WorkflowMultiBranchProject);
    }
}
