package com.groupon.jenkins.branchhistory;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.groupon.jenkins.dynamic.build.DbBackedBuild;
import com.groupon.jenkins.dynamic.build.DbBackedProject;
import com.groupon.jenkins.dynamic.build.DynamicProject;
import com.groupon.jenkins.dynamic.build.repository.DynamicBuildRepository;
import hudson.widgets.HistoryWidget;

import java.util.ArrayList;
import java.util.ListIterator;

class BranchHistoryWidgetModel<T extends DbBackedBuild> {

	private final String branch;
	private final DynamicBuildRepository dynamicBuildRepository;
	private final DynamicProject owner;

	public BranchHistoryWidgetModel(DynamicProject owner, DynamicBuildRepository dynamicBuildRepository, String branch) {
		this.owner = owner;
		this.dynamicBuildRepository = dynamicBuildRepository;
		this.branch = branch;
	}

	public Iterable<T> getBuildsAfter(String n) {
		return filterSkipped(isMyBuilds() ? dynamicBuildRepository.<T> getCurrentUserBuildsGreaterThan((DbBackedProject) owner, n) : dynamicBuildRepository.<T> getBuildGreaterThan((DbBackedProject) owner, n, branch));
	}

	public Iterable<T> getBaseList() {
		return filterSkipped(isMyBuilds() ? dynamicBuildRepository.<T> getCurrentUserBuilds(((DbBackedProject) owner), BranchHistoryWidget.BUILD_COUNT) : dynamicBuildRepository.<T> getLast((DbBackedProject) owner, BranchHistoryWidget.BUILD_COUNT, branch));
	}

	public OffsetBuild<T> getNextBuildToFetch(Iterable<T> builds, HistoryWidget.Adapter<? super T> adapter) {
		ArrayList<T> list = Lists.newArrayList(builds);
		if (!list.isEmpty()) {
			ListIterator<T> listIterator = list.listIterator(list.size());

			while (listIterator.hasPrevious()) {
				T record = listIterator.previous();
				if (adapter.isBuilding(record))
					return new OffsetBuild<T>(record, 0);
			}

			return new OffsetBuild(list.get(0), 1);
		}
		return new OffsetBuild<T>(null, 0);
	}

	private Iterable<T> filterSkipped(Iterable<T> builds) {
		return Iterables.filter(builds, new Predicate<T>() {
			@Override
			public boolean apply(T build) {
				return !build.isSkipped();
			}
		});
	}

	private boolean isMyBuilds() {
		return BranchHistoryWidget.MY_BUILDS_BRANCH.equals(this.branch);
	}

	protected static class OffsetBuild<T> {
		private final T t;
		private final int offset;

		public OffsetBuild(T t, int offset) {
			this.t = t;
			this.offset = offset;
		}

		public String getBuildNumber(HistoryWidget.Adapter<? super T> adapter) {
			if (t == null) {
				return "1";
			}
			return String.valueOf(Integer.valueOf(adapter.getKey(t)) + offset);
		}
	}
}