/*
The MIT License (MIT)

Copyright (c) 2014, Groupon, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
 */
package com.groupon.jenkins.dynamic.build;

import hudson.EnvVars;
import hudson.Util;
import hudson.model.Build;
import hudson.model.BuildListener;
import hudson.model.Result;
import hudson.model.TaskListener;
import hudson.model.AbstractProject;
import hudson.model.Computer;
import hudson.model.Executor;
import hudson.model.Queue.Executable;
import hudson.model.Run;
import hudson.model.User;
import hudson.scm.ChangeLogSet;
import hudson.scm.ChangeLogSet.Entry;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import jenkins.model.Jenkins;

import org.kohsuke.stapler.export.Exported;
import org.springframework.util.ReflectionUtils;

import com.groupon.jenkins.dynamic.build.cause.BuildCause;
import com.groupon.jenkins.dynamic.build.repository.DynamicBuildRepository;
import com.groupon.jenkins.github.GitBranch;

public abstract class DbBackedBuild<P extends DbBackedProject<P, B>, B extends DbBackedBuild<P, B>> extends Build<P, B> {
	private static final Logger LOGGER = Logger.getLogger(DbBackedBuild.class.getName());

	protected DbBackedBuild(P project) throws IOException {
		super(project);
	}

	public DbBackedBuild(P project, File buildDir) throws IOException {
		super(project, buildDir);
	}

	public DbBackedBuild(P job, Calendar timestamp) {
		super(job, timestamp);
	}

	@Override
	public synchronized void save() throws IOException {
		LOGGER.info("saving build:" + getName() + ": " + getNumber());
		Map<String, Object> attributes = getBuildAttributesForDb();
		new DynamicBuildRepository().save(this.getProject(), getNumber(), attributes);
	}

	@Override
	protected void onEndBuilding() {
		super.onEndBuilding();
		try {
			this.save();
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	protected Map<String, Object> getBuildAttributesForDb() {
		String buildXml = Run.XSTREAM.toXML(this);

		Object state = null;
		try {
			Field field = Run.class.getDeclaredField("state");
			field.setAccessible(true);
			state = ReflectionUtils.getField(field, this);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}

		Result buildResult = getResult() == null ? Result.NOT_BUILT : getResult();
		Map<String, Object> attributes = new HashMap<String, Object>();
		attributes.put("xml", buildXml);
		attributes.put("parent", project.getId());
		attributes.put("state", state.toString());
		attributes.put("result", buildResult.toString());
		attributes.put("number", getNumber());
		attributes.put("branch", getCurrentBranch().toString());
		attributes.put("pusher", getPusher());
		attributes.put("sha", getSha());
		attributes.put("built_on", getBuiltOnStr());
		attributes.put("last_updated", getTime());
		return attributes;
	}

	public void restoreFromDb(AbstractProject project, Map<String, Object> input) {
		String state = ((String) input.get("state"));
		Date date = ((Date) input.get("last_updated"));
		setField(project, "project");
		setField(date.getTime(), "timestamp");
		setField(getState(state), "state");
	}

	public Object getState(String state) {
		try {
			return Enum.<Enum> valueOf((Class<Enum>) Class.forName("hudson.model.Run$State"), state);
		} catch (ClassNotFoundException e) {
			throw new RuntimeException(e);
		}
	}

	@Override
	public boolean hasParticipant(User user) {
		return false;
	}

	@Override
	public B getNextBuild() {
		return getParent().getBuildByNumber(getNumber() + 1);
	}

	@Override
	public B getPreviousBuild() {
		return null;
	}

	private void setField(Object project, String fieldStr) {
		Field field;
		try {
			field = Run.class.getDeclaredField(fieldStr);
			field.setAccessible(true);
			field.set(this, project);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	@Override
	@Exported
	public Executor getExecutor() {
		Jenkins jenkins = Jenkins.getInstance();
		if (jenkins == null) {
			return null;
		}
		for (Computer computer : jenkins.getComputers()) {
			for (Executor executor : computer.getExecutors()) {
				if (isCurrent(executor)) {
					return executor;
				}
			}
		}
		return null;
	}

	private boolean isCurrent(Executor executor) {
		Executable currentExecutable = executor.getCurrentExecutable();
		return currentExecutable != null && currentExecutable instanceof DbBackedBuild && this.equals(currentExecutable);
	}

	@Override
	public Executor getOneOffExecutor() {
		for (Computer c : Jenkins.getInstance().getComputers()) {
			for (Executor e : c.getOneOffExecutors()) {
				if (isCurrent(e))
					return e;
			}
		}
		return null;
	}

	@Override
	@Exported
	public ChangeLogSet<? extends Entry> getChangeSet() {
		return new GithubChangeLogSet(this, getCause().getChangeLogEntries());
	}

	public String getName() {
		return getParent().getName();
	}

	@Override
	@Exported
	public Set<User> getCulprits() {
		return new HashSet<User>();
	}

	public GitBranch getCurrentBranch() {
		return getCause().getBranch();
	}

	public String getDisplayTime() {
		return Util.getPastTimeString(System.currentTimeMillis() - getTimestamp().getTimeInMillis()) + " ago";
	}

	public abstract BuildCause getCause();

	@Override
	public EnvVars getEnvironment(TaskListener log) throws IOException, InterruptedException {
		EnvVars envVars = getJenkinsEnvVariables(log);
		envVars.putAll(getDotCiEnvVars(envVars));
		return envVars;
	}

	public Map<String, String> getDotCiEnvVars(EnvVars jenkinsEnvVars) {
		Map<String, String> envVars = new HashMap<String, String>();
		envVars.put("DOTCI_BRANCH", jenkinsEnvVars.get("BRANCH"));
		envVars.put("DOTCI", "true");
		envVars.put("CI", "true");
		if (getCause() != null) {
			envVars.putAll(getCause().getEnvVars());
		}
		return envVars;
	}

	public String getBuildId() {
		return getProject().getId() + getId();
	}

	public EnvVars getJenkinsEnvVariables(TaskListener log) {
		try {
			return super.getEnvironment(log);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	public boolean isSkipped() {
		return getAction(SkippedBuildAction.class) != null;
	}

	@Override
	public String getWhyKeepLog() {
		return null;
	}

	public Run getPreviousFinishedBuildOfSameBranch(BuildListener listener) {
		return new DynamicBuildRepository().getPreviousFinishedBuildOfSameBranch(this, getCurrentBranch().toString());
	}

	public String getPusher() {
		return getCause() == null ? null : getCause().getPusher();
	}

	public abstract String getSha();

}