package com.atlassian.plugin.osgi.internal.hook.rest;

import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.osgi.framework.Bundle;
import org.osgi.framework.wiring.BundleCapability;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URL;
import java.util.function.Predicate;

import static com.atlassian.plugin.osgi.hook.rest.RestVersionUtils.isCapabilityWithMajorVersion;

public class JaxRsFilterFactory {

    private static final Logger LOGGER = LoggerFactory.getLogger(JaxRsFilterFactory.class);

    public static final Predicate<BundleCapability> FILTER_REST_V1 = bundleCapability -> isCapabilityWithMajorVersion(bundleCapability, 1);
    public static final Predicate<BundleCapability> FILTER_REST_V2 = bundleCapability -> isCapabilityWithMajorVersion(bundleCapability, 2);

    /**
     * Checks if bundle defines `rest-migration` tag with 'rest-v2' element. If at least one module is defined returned
     *
     * @param bundle
     * @return
     */
    public Predicate<BundleCapability> getFilter(Bundle bundle) {
        URL pluginDescriptor = bundle.getResource("atlassian-plugin.xml");
        Document read = null;
        try {
            read = new SAXReader().read(pluginDescriptor);

            return getFilter(read);
        } catch (DocumentException e) {
            LOGGER.warn(String.format("Cannot parse plugin descriptor for bundle %s; no filtering of JAX-RS package", bundle.getSymbolicName()), e);
            return ignored -> true;
        }
    }

    public Predicate<BundleCapability> getFilter(Document pluginXml) {
        Element restMigration = pluginXml.getRootElement().element("rest-migration");
        boolean hasRestV2 = restMigration != null && restMigration.element("rest-v2") != null;
        return hasRestV2 ? FILTER_REST_V2 : FILTER_REST_V1;
    }
}
