package com.atlassian.plugin.osgi.hook;

import com.atlassian.plugin.osgi.util.OsgiHeaderUtil;
import org.osgi.framework.hooks.resolver.ResolverHook;
import org.osgi.framework.wiring.BundleCapability;
import org.osgi.framework.wiring.BundleRequirement;
import org.osgi.framework.wiring.BundleRevision;

import java.util.Collection;
import java.util.Objects;
import java.util.Set;

/**
 * Implementation based on stash DMZ impl:
 * <a href="https://stash.atlassian.com/projects/STASH/repos/stash/browse/platform/src/main/java/com/atlassian/stash/internal/osgi/DmzResolverHook.java">DmzResolverHook</a>
 */
public class DmzResolverHook implements ResolverHook {

    static final String ATTR_WIRING_PACKAGE = "osgi.wiring.package";

    private final Set<String> bundledPluginKeys;
    private final Set<String> dmzPackages;

    public DmzResolverHook(Set<String> bundledPluginKeys, Set<String> dmzPackages) {
        Objects.requireNonNull(bundledPluginKeys, "Bundled plugin keys required.");
        Objects.requireNonNull(dmzPackages, "DMZ packages required.");
        this.bundledPluginKeys = bundledPluginKeys;
        this.dmzPackages = dmzPackages;
    }

    @Override
    public void end() {
    }

    @Override
    public void filterMatches(BundleRequirement requirement, Collection<BundleCapability> candidates) {
        String pluginKey = OsgiHeaderUtil.getPluginKey(requirement.getRevision().getBundle());
        if (!bundledPluginKeys.contains(pluginKey)) {
            candidates.removeIf(this::isDmzPackage);
        }
    }

    @Override
    public void filterResolvable(Collection<BundleRevision> candidates) {
    }

    @Override
    public void filterSingletonCollisions(BundleCapability singleton, Collection<BundleCapability> collisions) {
    }

    private boolean isDmzPackage(BundleCapability candidate) {
        Object wiringPackageAtt = candidate.getAttributes().get(ATTR_WIRING_PACKAGE);
        if (wiringPackageAtt instanceof String) {
            String wiringPackage = (String) wiringPackageAtt;
            return dmzPackages.stream().anyMatch(wiringPackage::startsWith);
        }
        return false;
    }
}

