package com.atlassian.plugin.osgi.container;

import javax.servlet.ServletContext;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Contains configuration for the package scanning algorithm that scans your classpath to determine which packages
 * and versions to export to OSGi. Jar and package includes/excludes, and packages for the package version map can
 * either be simple names or wildcard patterns, where the "*" character will match any character.
 *
 * <p>
 * Includes and excludes are matched so that only includes are, well, included, but if you need to filter a few out
 * of that set, the exclude patterns will be removed.
 */
public interface PackageScannerConfiguration {
    /**
     * @return The jar patterns to include
     */
    List<String> getJarIncludes();

    /**
     * @return The jar patterns to exclude
     */
    List<String> getJarExcludes();

    /**
     * @return The package patterns to include
     */
    List<String> getPackageIncludes();

    /**
     * @return The package patterns to exclude
     */
    List<String> getPackageExcludes();

    /**
     * @return A map of package patterns and their versions
     */
    Map<String, String> getPackageVersions();

    /**
     * @return The current host application version number. Used as a caching key for scanned data.
     * @since 2.2.0
     */
    String getCurrentHostVersion();

    /**
     * @return The servlet context to use to scan for jars, in case the classloader scanning fails
     */
    ServletContext getServletContext();

    /**
     * @return List all packages that should be available to bundled plugins, but not third-party plugins.
     * <p>
     * Example:
     * <p>
     * com.atlassian.refapp.dmz
     * <p>
     */
    Set<String> getOsgiDmzPackages();

    /**
     * <p>
     * Application bundled internal plugin is determined by atlassian plugin key.
     * <p>
     * Atlassian plugin key can be found in atlassian plugin pom.xml file.
     * <p>
     * Each pom.xml has atlassian.plugin.key property.
     * <p>
     * Example:
     * <p>
     * &lt;atlassian.plugin.key&gt;com.atlassian.refapp.ao&lt;/atlassian.plugin.key&gt;
     * <p>
     * &lt;atlassian.plugin.key&gt;com.atlassian.refapp.auth&lt;/atlassian.plugin.key&gt;
     * <p>
     * @return Application bundled internal plugins (list of atlassian plugin keys). These plugins have DMZ access.
     * <p>
     * Example:
     * <p>
     * com.atlassian.refapp.ao
     * <p>
     * com.atlassian.refapp.auth
     */
    Set<String> getApplicationBundledInternalPlugins();
}
