package com.atlassian.plugin.osgi.factory;

import static com.atlassian.plugin.osgi.util.OsgiHeaderUtil.extractOsgiPluginInformation;
import static com.atlassian.plugin.osgi.util.OsgiHeaderUtil.getAttributeWithoutValidation;
import static com.atlassian.plugin.osgi.util.OsgiHeaderUtil.getManifest;
import static com.google.common.base.Preconditions.checkNotNull;

import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.jar.Manifest;

import org.osgi.framework.Bundle;
import org.osgi.framework.BundleEvent;
import org.osgi.framework.BundleException;
import org.osgi.framework.Constants;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.SynchronousBundleListener;
import org.osgi.framework.wiring.BundleRevision;
import org.osgi.framework.wiring.BundleRevisions;
import org.osgi.framework.wiring.BundleWire;
import org.osgi.framework.wiring.BundleWiring;
import org.osgi.service.packageadmin.PackageAdmin;
import org.osgi.util.tracker.ServiceTracker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.atlassian.plugin.IllegalPluginStateException;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginArtifact;
import com.atlassian.plugin.PluginArtifactBackedPlugin;
import com.atlassian.plugin.PluginException;
import com.atlassian.plugin.PluginState;
import com.atlassian.plugin.event.events.PluginRefreshedEvent;
import com.atlassian.plugin.impl.AbstractPlugin;
import com.atlassian.plugin.module.ContainerAccessor;
import com.atlassian.plugin.module.ContainerManagedPlugin;
import com.atlassian.plugin.osgi.container.OsgiContainerException;
import com.atlassian.plugin.osgi.container.OsgiContainerManager;
import com.atlassian.plugin.osgi.util.BundleClassLoaderAccessor;
import com.atlassian.plugin.osgi.util.OsgiHeaderUtil;
import com.atlassian.plugin.util.resource.AlternativeDirectoryResourceLoader;
import com.atlassian.util.concurrent.NotNull;
import com.atlassian.util.concurrent.Nullable;

/**
 * Usual atlassian plugin that has no Spring context
 */
public final class OsgiLightweightPlugin extends AbstractPlugin implements PluginArtifactBackedPlugin, ContainerManagedPlugin, SynchronousBundleListener, Plugin.Resolvable
{
    private static final Logger log = LoggerFactory.getLogger(OsgiLightweightPlugin.class);

    /*
     * Underlaying storage with the plugin files. Usually JAR archive
     */
    private final @NotNull PluginArtifact pluginArtifact;
    
    /*
     * ???
     */
    private final Date dateLoaded;

    /*
     * The OSGi container manager to install plugin into.
     * That is a cached value that is required only until plugin is actually installed. Once 
     * that happens field will be nulled (see installInternal) 
     */
    private @Nullable OsgiContainerManager osgiContainerManager;
    
    /*
     * OSGi bundle that corresponds to the Plugin. That field is null until actual Bundle
     * created, see installInternal.
     * 
     * Note that is accessed from different threads
     */
    private volatile @Nullable Bundle bundle;
    
    /*
     * Classloader to be used for the Plugin which is usually is a OSGi Bundle class loader. 
     * Field stays null until actual bundle created, see installInternal. 
     */
    private @Nullable ClassLoader bundleClassLoader;
    
    /*
     * Service exported by bundle to provide access to internal IoC.
     * Field initialized in a lazy manner. 
     */
    private @Nullable ServiceTracker<ContainerAccessor, ContainerAccessor> containerAccessor;
    
    /*
     * Service to be used to calculate bundle wiring and extract plugin dependencies in terms of
     * Atlassian Plugins Framework
     */
    private @Nullable ServiceTracker<PackageAdmin, PackageAdmin> pkgAdminService;


    /*
     * Notes on concurrency:
     * All methods with *Internal suffix are called from corresponding wrappers from the base class,
     * i.e installInternal from install. Wrapper call takes care about plugin state update which is effectively
     * volatile variable write (AtomicReference.set). That write is a guarantee of all internal state changes safe
     * publication. 
     * 
     * However methods are *not* thread safe, as there is no protection against race execution. In some cases
     * that won't lead to any problems except not optimal resource management but special care should be taken in
     * each particular scenario to guarantee data structures valid states
     * 
     * Make sure that is any of *Internal method returns PENDING state it takes care to make changes safely 
     * published in terms of concurrency
     */

    /**
     * Create a plugin wrapper which installs the bundle when the plugin is installed.
     *
     * @param osgiContainerManager the container to install into when the plugin is installed.
     * @param pluginKey The plugin key.
     * @param pluginArtifact The The plugin artifact to install.
     */
    @SuppressWarnings("unchecked")
    public OsgiLightweightPlugin(final OsgiContainerManager osgiContainerManager, String pluginKey, PluginArtifact pluginArtifact)
    {
        this.dateLoaded = new Date();
        this.pluginArtifact = checkNotNull(pluginArtifact);
        this.osgiContainerManager = checkNotNull(osgiContainerManager);
        this.pkgAdminService = osgiContainerManager.getServiceTracker(PackageAdmin.class.getName());

        // Leave bundle and bundleClassLoader null until we are installed.
        this.bundle = null;
        this.bundleClassLoader = null;

        Manifest manifest = getManifest(pluginArtifact);
        
        // Initialize meta information about the plugin
        setKey(pluginKey);
        setPluginsVersion(2);
        setSystemPlugin(false);
        setName(getAttributeWithoutValidation(manifest, Constants.BUNDLE_NAME));
        setPluginInformation(extractOsgiPluginInformation(manifest, false));
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.impl.AbstractPlugin#getDateLoaded()
     */
    @Override
    public Date getDateLoaded()
    {
        return dateLoaded;
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.impl.AbstractPlugin#getDateInstalled()
     */
    @Override
    public Date getDateInstalled()
    {
        long date = getPluginArtifact().toFile().lastModified();
        if (date == 0)
        {
            date = getDateLoaded().getTime();
        }
        return new Date(date);
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.Plugin#isUninstallable()
     */
    public boolean isUninstallable()
    {
        return true;
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.Plugin#isDeleteable()
     */
    public boolean isDeleteable()
    {
        return true;
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.Plugin#isDynamicallyLoaded()
     */
    public boolean isDynamicallyLoaded()
    {
        return true;
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.Plugin#loadClass(java.lang.String, java.lang.Class)
     */
    public <T> Class<T> loadClass(final String clazz, final Class<?> callingClass) throws ClassNotFoundException
    {
        return BundleClassLoaderAccessor.loadClass(getBundleOrFail(), clazz);
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.Plugin#getResource(java.lang.String)
     */
    public URL getResource(final String name)
    {
        return getBundleClassLoaderOrFail().getResource(name);
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.Plugin#getResourceAsStream(java.lang.String)
     */
    public InputStream getResourceAsStream(final String name)
    {
        return getBundleClassLoaderOrFail().getResourceAsStream(name);
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.Plugin.Resolvable#resolve()
     */
    @Override
    public void resolve()
    {
        PackageAdmin packageAdmin = pkgAdminService.getService();
        packageAdmin.resolveBundles(new Bundle[] { bundle });
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.impl.AbstractPlugin#installInternal()
     */
    @Override
    protected void installInternal() throws OsgiContainerException, IllegalPluginStateException
    {
        super.installInternal();
        log.debug("Installing OSGi bundled plugin '{}'", getKey());

        if (null != osgiContainerManager)
        {
            // We're pending installation, so install
            final File file = pluginArtifact.toFile();
            final boolean allowReference = PluginArtifact.AllowsReference.Default.allowsReference(pluginArtifact);
            bundle = OsgiContainerManager.AllowsReferenceInstall.Default.installBundle(osgiContainerManager, file, allowReference);
            bundleClassLoader = BundleClassLoaderAccessor.getClassLoader(bundle, new AlternativeDirectoryResourceLoader());
            osgiContainerManager.addBundleListener(this);
        }
        else if (null == bundle)
        {
            throw new IllegalPluginStateException("Cannot reuse instance for bundle '" + getKey() + "'");
        }
        // else this could be a reinstall or not, but we can't tell, so we let it slide
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.impl.AbstractPlugin#uninstallInternal()
     */
    @Override
    protected void uninstallInternal()
    {
        super.uninstallInternal();
        log.debug("Uninstalling OSGi bundled plugin '{}'", getKey());

        try
        {
            if (bundleIsUsable("uninstall"))
            {
                if (bundle.getState() != Bundle.UNINSTALLED)
                {
                    bundle.uninstall();
                }
                else
                {
                    // A previous uninstall aborted early ?
                    log.warn("Bundle '{}' already UNINSTALLED, but still held", getKey());
                }
                osgiContainerManager.removeBundleListener(this);
                bundle = null;
                bundleClassLoader = null;
                osgiContainerManager = null;
            }
        }
        catch (final BundleException e)
        {
            throw new PluginException(e);
        }
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.impl.AbstractPlugin#enableInternal()
     */
    @Override
    protected PluginState enableInternal()
    {
        super.enableInternal();
        log.debug("Enabling OSGi bundled plugin '{}'", getKey());
        try
        {
            if (bundleIsUsable("enable"))
            {
                if (bundle.getHeaders().get(Constants.FRAGMENT_HOST) == null)
                {
                    log.debug("Plugin '{}' bundle is NOT a fragment, starting.", getKey());
                    // TODO: Shouldn't we add listener in front of start to not loose some
                    // notifications? Or at lest to not create race condition. IT is necessary
                    // to be confirmed with OSGi specification
                    setPluginState(PluginState.ENABLING);
                    bundle.start();

                    // It is needs to give plugin time to launch Activator and register any services
                    // before to move on. During activator work, plugin might register ContainerAccessor
                    // service which will be used later during ModuleDescriptors activation
                    //
                    // ENABLED state will be setup by the listener
                    // ENABLING is setup explicitly BEFORE bundle actually start to avoid
                    // race condition between current & osgi activator threads
                    // PENDING returned to avooid external state managemnt
                    return PluginState.PENDING;
                }
                else
                {
                    log.debug("Plugin '{}' bundle is a fragment, not doing anything.", getKey());
                }
            }
            return PluginState.ENABLED;
        }
        catch (final BundleException e)
        {
            throw new PluginException(e);
        }
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.impl.AbstractPlugin#disableInternal()
     */
    @Override
    protected void disableInternal()
    {
        super.disableInternal();
        log.debug("Disabling OSGi bundled plugin '{}'", getKey());

        try
        {
            if (bundleIsUsable("disable"))
            {
                if (bundle.getState() == Bundle.ACTIVE)
                {
                    containerAccessor.close();
                    bundle.stop();
                }
                else
                {
                    log.warn("Cannot disable Bundle '{}', not ACTIVE", getKey());
                }
            }
        }
        catch (final BundleException e)
        {
            throw new PluginException(e);
        }
    }

    /* (non-Javadoc)
     * @see org.osgi.framework.BundleListener#bundleChanged(org.osgi.framework.BundleEvent)
     * 
     * Event bridge to synchronize Atlassian Plugin and OSGi Bundle lifecycles
     */
    @Override
    public void bundleChanged(@NotNull BundleEvent bundleEvent)
    {
        // If bundle is null, this == will be false. This silently ignores events after we're uninstalled,
        // but i am ok with that, because that's the prior behaviour, and we can preserve that for now.
        if (bundleEvent.getBundle() == bundle)
        {
            log.info("bundleChanged({}, {})", bundleEvent.getBundle().getSymbolicName(), bundleEvent.getType());
            if (bundleEvent.getType() == BundleEvent.STOPPING)
            {
                log.info("disabled({}): from {}", bundleEvent.getBundle().getSymbolicName(), getPluginState());
                setPluginState(PluginState.DISABLED);
            }
            else if (bundleEvent.getType() == BundleEvent.STARTED)
            {
                log.info("enabled({}): from {}", bundleEvent.getBundle().getSymbolicName(), getPluginState());
                // Bundle was enabled, so it is time to take care of ContainerAccessor
                this.containerAccessor = new ServiceTracker<>(
                        bundle.getBundleContext(),
                        ContainerAccessor.class,
                        null
                );
                this.containerAccessor.open();
                setPluginState(PluginState.ENABLED);
            }
        }
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.Plugin#getClassLoader()
     */
    public ClassLoader getClassLoader()
    {
        return getBundleClassLoaderOrFail();
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.PluginArtifactBackedPlugin#getPluginArtifact()
     */
    public PluginArtifact getPluginArtifact()
    {
        return pluginArtifact;
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.module.ContainerManagedPlugin#getContainerAccessor()
     */
    @Override
    public ContainerAccessor getContainerAccessor()
    {
        ContainerAccessor result = containerAccessor.getService();
        if (result == null) {
            result = ContainerAccessor.NULL;
        }
        return result;
    }

    
    /* (non-Javadoc)
     * @see com.atlassian.plugin.impl.AbstractPlugin#getRequiredPlugins()
     */
    @Override
    public Set<String> getRequiredPlugins() throws IllegalPluginStateException
    {
        PackageAdmin pkgAdmin = pkgAdminService.getService();
        if (pkgAdmin == null) {
            log.debug("Failed to find PackageAdmin service");
            return Collections.emptySet();
        }
        
        // A bundle must move from INSTALLED to RESOLVED before we can get its import
        if (bundle.getState() == Bundle.INSTALLED)
        {
            log.debug("Bundle is in INSTALLED for {}", bundle.getSymbolicName());
            if (!pkgAdmin.resolveBundles(new Bundle[] { bundle }))
            {
                // The likely cause of this is installing a plugin without installing everything that it requires.
                log.error("Cannot determine required plugins, cannot resolve bundle '{}'", bundle.getSymbolicName());
                return Collections.emptySet();
            }
            log.debug("Bundle state is now {}", bundle.getState());
        }
        
        // Convert found wires into corresponding atlassian plugins
        Set<String> keys = new HashSet<String>();
        if (bundle instanceof BundleRevisions)
        {
            for (BundleRevision bundleRevision : ((BundleRevisions) bundle).getRevisions())
            {
                for (BundleWire requiredWire : bundleRevision.getWiring().getRequiredWires(null))
                {
                    keys.add(OsgiHeaderUtil.getPluginKey(requiredWire.getProviderWiring().getBundle()));
                }
            }
        }
        return keys;
    }

    /*
     * Return current plugin state as string human readable description
     */
    private String getInstallationStateExplanation()
    {
        return (null != osgiContainerManager) ? "not yet installed" : "already uninstalled";
    }

    /*
     * Tests if current plugin is ready to perform given operation and issue LOG warning if
     * not as it means logical bug in plugin lifecycle sequence
     */
    private boolean bundleIsUsable(final String task)
    {
        if (null != bundle)
        {
            return true;
        }
        else
        {
            final String why = getInstallationStateExplanation();
            log.warn("Cannot {} {} bundle '{}'", new Object[] { task, why, getKey()});
            return false;
        }
    }

    /*
     * Protect against nulls
     */
    private <T> T getOrFail(final T what, final String name) throws PluginException
    {
        if (null == what)
        {
            throw new IllegalPluginStateException("Cannot use " + name + " of " + getInstallationStateExplanation() + " '"
                    + getKey() + "' from '" + pluginArtifact + "'");
        }
        else
        {
            return what;
        }
    }

    private Bundle getBundleOrFail() throws PluginException
    {
        return getOrFail(bundle, "bundle");
    }

    private ClassLoader getBundleClassLoaderOrFail() throws PluginException
    {
        return getOrFail(bundleClassLoader, "bundleClassLoader");
    }
}
