package com.atlassian.plugin.osgi.factory;

import java.io.IOException;
import java.io.InputStream;
import java.util.jar.Manifest;

import com.atlassian.plugin.Application;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.ModuleDescriptorFactory;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.PluginArtifact;
import com.atlassian.plugin.PluginParseException;
import com.atlassian.plugin.factories.AbstractPluginFactory;
import com.atlassian.plugin.impl.UnloadablePlugin;
import com.atlassian.plugin.osgi.container.OsgiContainerManager;
import com.atlassian.plugin.osgi.factory.transform.PluginTransformationException;
import com.atlassian.plugin.parsers.DescriptorParser;

import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Ranges;

import org.apache.commons.io.IOUtils;
import org.osgi.framework.Constants;
import org.osgi.util.tracker.ServiceTracker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.atlassian.plugin.osgi.util.OsgiHeaderUtil.getManifest;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Plugin deployer that deploys atlassian plugins without Spring context, i.e to be loaded by the Factory bundle
 * should be non Spring powered.
 * 
 * That type used for plugins which are managing IoC containers by them self, like OSGi Blueprint, external Spring, e.t.c
 */
public final class OsgiLightweightPluginFactory extends AbstractPluginFactory
{
    private static final Logger log = LoggerFactory.getLogger(OsgiLightweightPluginFactory.class);

    /*
     * OSGi container to install plugin to 
     */
    private final OsgiContainerManager osgi;

    /*
     * Atlassian plugin descriptor filename 
     */
    private final String pluginDescriptorFileName;

    /*
     * Build to create plugin specific ModuleFactories needs to parse descriptors (atlassian-plugin.xml)
     */
    private final OsgiChainedModuleDescriptorFactoryCreator osgiChainedModuleDescriptorFactoryCreator;

    /**
     * Builds new factory with default plugin descriptor filename. That is equivalent to the call {@link
     * OsgiLightweightPluginFactory(PluginAccessor.Descriptor.FILENAME, osgi)}
     *
     * @param osgi container to install plugin in
     */
    public OsgiLightweightPluginFactory(OsgiContainerManager osgi)
    {
        this(PluginAccessor.Descriptor.FILENAME, osgi);
    }

    /**
     * Builds new factory with specified plugin descriptor filename.
     *
     * @param osgi container to install plugin in
     */
    public OsgiLightweightPluginFactory(String pluginDescriptorFileName, OsgiContainerManager osgi)
    {
        super(new OsgiPluginXmlDescriptorParserFactory(), ImmutableSet.<Application>of());
        this.pluginDescriptorFileName = checkNotNull(pluginDescriptorFileName);
        this.osgi = checkNotNull(osgi, "The osgi container is required");
        this.osgiChainedModuleDescriptorFactoryCreator = new OsgiChainedModuleDescriptorFactoryCreator(new OsgiChainedModuleDescriptorFactoryCreator.ServiceTrackerFactory()
        {
            public ServiceTracker create(String className)
            {
                return OsgiLightweightPluginFactory.this.osgi.getServiceTracker(className);
            }
        });
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.factories.AbstractPluginFactory#getDescriptorInputStream(com.atlassian.plugin.PluginArtifact)
     */
    @Override
    protected InputStream getDescriptorInputStream(PluginArtifact pluginArtifact)
    {
        return pluginArtifact.getResourceAsStream(pluginDescriptorFileName);
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.factories.AbstractPluginFactory#isValidPluginsVersion()
     */
    @Override
    protected Predicate<Integer> isValidPluginsVersion()
    {
        return Ranges.atLeast(Plugin.VERSION_2);
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.factories.AbstractPluginFactory#canCreate(com.atlassian.plugin.PluginArtifact)
     * 
     * Checks if factory could creates plugin from the given artifact. That could be if bundle is not Spring
     * powered but has atlassian-plugin.xml. Note that some parts of plugin descriptor, such as <component> &
     * <component-import>, relays on Spring for work, so once defined in descriptor made bundle to be Spring powered 
     */
    public String canCreate(final PluginArtifact pluginArtifact) throws PluginParseException
    {
        checkNotNull(pluginArtifact, "The plugin artifact is required");

        InputStream descriptorStream = null;
        try
        {
            // Check #1 - atlassian plugin descriptor is there
            descriptorStream = getDescriptorInputStream(pluginArtifact);
            if (null != descriptorStream)
            {
                // Check #2 - descriptor doesn't contains any Spring powered instructions
                // TODO: implements

                // Check #3 - default spring xml files location
                // TODO: better check - if any files actual present & case insensitive
                // maybe copy check from gemini
                if (pluginArtifact.doesResourceExist("META-INF/spring"))
                {
                    // Bundle is Spring-powered
                    return null;
                }

                // Check #4 - Manifest doesn't have Spring header  & has atlassian plugin key
                // TODO: separate checks && LOG cases as OsgiPluginFactory does
                final Manifest manifest = getManifest(pluginArtifact);
                if (
                        (null != manifest) && (
                                null != manifest.getMainAttributes().getValue(OsgiPlugin.ATLASSIAN_PLUGIN_KEY) &&
                                        null != manifest.getMainAttributes().getValue(Constants.BUNDLE_VERSION) &&
                                        // Constant is defined in gemini which is _not_ in dependencies list
                                        null == manifest.getMainAttributes().getValue("Spring-Context")
                        )
                        )
                {
                    return manifest.getMainAttributes().getValue(OsgiPlugin.ATLASSIAN_PLUGIN_KEY);
                }
                else
                {
                    // Bundle is Spring-powered or has no manifest or has no Atlassian key or has no version
                    return null;
                }
            }
            else
            {
                // It is not an Atlassian Plugin
                return null;
            }
        }
        finally
        {
            IOUtils.closeQuietly(descriptorStream);
        }
    }

    /* (non-Javadoc)
     * @see com.atlassian.plugin.factories.PluginFactory#create(com.atlassian.plugin.PluginArtifact, com.atlassian.plugin.ModuleDescriptorFactory)
     */
    @Override
    public Plugin create(final PluginArtifact pluginArtifact, final ModuleDescriptorFactory moduleDescriptorFactory)
    {
        checkNotNull(pluginArtifact, "The plugin deployment unit is required");
        checkNotNull(moduleDescriptorFactory, "The module descriptor factory is required");

        Plugin plugin = null;
        try (InputStream pluginDescriptor = pluginArtifact.getResourceAsStream(pluginDescriptorFileName))
        {
            // Plugin specific module factory to allow plugin internal resources usage
            ModuleDescriptorFactory combinedFactory = osgiChainedModuleDescriptorFactoryCreator.create(new OsgiChainedModuleDescriptorFactoryCreator.ResourceLocator()
            {
                public boolean doesResourceExist(String name)
                {
                    return pluginArtifact.doesResourceExist(name);
                }
            }, moduleDescriptorFactory);


            // Parse descriptor & configure plugin
            DescriptorParser parser = descriptorParserFactory.getInstance(pluginDescriptor, applications);
            Plugin osgiPlugin = new OsgiLightweightPlugin(osgi, parser.getKey(), pluginArtifact);
            plugin = parser.configurePlugin(combinedFactory, osgiPlugin);
        }
        catch (PluginTransformationException | IOException ex)
        {
            log.error("Unable to load plugin: " + pluginArtifact.toFile(), ex);
            plugin = new UnloadablePlugin("Unable to load plugin: " + ex.getMessage());
        }
        return plugin;
    }

    @Override
    public ModuleDescriptor<?> createModule(
            final Plugin plugin, final String moduleType, final ModuleDescriptorFactory moduleDescriptorFactory)
    {
        // Not supported yet
        return null;
    }
}
