package com.atlassian.plugin.osgi.factory;

import com.atlassian.plugin.IllegalPluginStateException;
import com.atlassian.plugin.PluginArtifact;
import com.atlassian.plugin.PluginArtifactBackedPlugin;
import com.atlassian.plugin.PluginException;
import com.atlassian.plugin.PluginInformation;
import com.atlassian.plugin.PluginPermission;
import com.atlassian.plugin.PluginState;
import com.atlassian.plugin.impl.AbstractPlugin;
import com.atlassian.plugin.osgi.container.OsgiContainerException;
import com.atlassian.plugin.osgi.container.OsgiContainerManager;
import com.atlassian.plugin.osgi.util.BundleClassLoaderAccessor;
import com.atlassian.plugin.util.resource.AlternativeDirectoryResourceLoader;
import com.google.common.collect.ImmutableSet;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleEvent;
import org.osgi.framework.BundleException;
import org.osgi.framework.Constants;
import org.osgi.framework.SynchronousBundleListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.util.Date;
import java.util.jar.Manifest;

import static com.atlassian.plugin.osgi.util.OsgiHeaderUtil.extractOsgiPluginInformation;
import static com.atlassian.plugin.osgi.util.OsgiHeaderUtil.getAttributeWithoutValidation;
import static com.atlassian.plugin.osgi.util.OsgiHeaderUtil.getManifest;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Plugin that wraps an OSGi bundle that has no plugin descriptor.
 */
public final class OsgiBundlePlugin extends AbstractPlugin implements PluginArtifactBackedPlugin
{
    private static final Logger log = LoggerFactory.getLogger(OsgiBundlePlugin.class);

    private final PluginArtifact pluginArtifact;
    private final SynchronousBundleListener bundleStartStopListener;
    private final Date dateLoaded;

    /** The OSGi container manager, which will be null after installInternal is called. */
    private OsgiContainerManager osgiContainerManager;
    /** The OSGi bundle, which will be null until installInternal is called. */
    private Bundle bundle;
    /** The ClassLoader for the OSGi bundle, which will be null until installInternal is called. */
    private ClassLoader bundleClassLoader;

    private OsgiBundlePlugin(final String pluginKey, final PluginArtifact pluginArtifact)
    {
        this.pluginArtifact = checkNotNull(pluginArtifact);
        this.bundleStartStopListener = new SynchronousBundleListener()
        {
            public void bundleChanged(final BundleEvent bundleEvent)
            {
                // If bundle is null, this == will be false. This silently ignores events after we're uninstalled,
                // but i am ok with that, because that's the prior behaviour, and we can preserve that for now.
                if (bundleEvent.getBundle() == bundle)
                {
                    if (bundleEvent.getType() == BundleEvent.STOPPING)
                    {
                        setPluginState(PluginState.DISABLED);
                    }
                    else if (bundleEvent.getType() == BundleEvent.STARTED)
                    {
                        setPluginState(PluginState.ENABLED);
                    }
                }
            }
        };
        this.dateLoaded = new Date();
        setPluginsVersion(2);
        setKey(pluginKey);
        setSystemPlugin(false);
    }

    /**
     * Create a plugin wrapper for an already installed Bundle.
     * @param bundle The installed Bundle.
     * @param pluginKey The plugin key.
     * @param pluginArtifact The plugin artifact that was installed.
     * @deprecated since 3.0.23, use {@link #OsgiBundlePlugin(OsgiContainerManager, String, PluginArtifact)}
     */
    @Deprecated
    public OsgiBundlePlugin(final Bundle bundle, final String pluginKey, final PluginArtifact pluginArtifact)
    {
        this(pluginKey, pluginArtifact);
        // Leave container null, as we're already installed
        this.bundle = checkNotNull(bundle);
        this.bundleClassLoader = BundleClassLoaderAccessor.getClassLoader(bundle, new AlternativeDirectoryResourceLoader());

        // It feels like this should be a call to OsgiHeaderUtil.extractOsgiPluginInformation, but Bundle.getHeaders is the square
        // peg Dictionary to extractOsgiPluginInformation's round holed Manifest. Rather than switching this code to parse the jar
        // again instead, I'm leaving it as is but deprecated.
        final PluginInformation pluginInformation = new PluginInformation();
        pluginInformation.setDescription((String) bundle.getHeaders().get(Constants.BUNDLE_DESCRIPTION));
        pluginInformation.setVersion((String) bundle.getHeaders().get(Constants.BUNDLE_VERSION));
        pluginInformation.setVendorName((String) bundle.getHeaders().get(Constants.BUNDLE_VENDOR));
        pluginInformation.setPermissions(ImmutableSet.of(PluginPermission.EXECUTE_JAVA));
        setPluginInformation(pluginInformation);

        setName((String) bundle.getHeaders().get(Constants.BUNDLE_NAME));
        setSystemPlugin(false);
    }

    /**
     * Create a plugin wrapper which installs the bundle when the plugin is installed.
     *
     * @param osgiContainerManager the container to install into when the plugin is installed.
     * @param pluginKey The plugin key.
     * @param pluginArtifact The The plugin artifact to install.
     */
    public OsgiBundlePlugin(final OsgiContainerManager osgiContainerManager,
            final String pluginKey,
            final PluginArtifact pluginArtifact)
    {
        this(pluginKey, pluginArtifact);
        this.osgiContainerManager = checkNotNull(osgiContainerManager);
        // Leave bundle and bundleClassLoader null until we are installed.

        final Manifest manifest = getManifest(pluginArtifact);
        if (null != manifest)
        {
            setName(getAttributeWithoutValidation(manifest, Constants.BUNDLE_NAME));
            // The next false is because at the OSGi level, Bundle-Version is not required.
            setPluginInformation(extractOsgiPluginInformation(manifest, false));
        }
        // else this will get flagged as a bad jar, because it's not a bundle, later, and we can let this through
    }

    @Override
    public Date getDateLoaded()
    {
        return dateLoaded;
    }

    @Override
    public Date getDateInstalled()
    {
        long date = getPluginArtifact().toFile().lastModified();
        if (date == 0)
        {
            date = getDateLoaded().getTime();
        }
        return new Date(date);
    }

    public boolean isUninstallable()
    {
        return true;
    }

    public boolean isDeleteable()
    {
        return true;
    }

    public boolean isDynamicallyLoaded()
    {
        return true;
    }

    public <T> Class<T> loadClass(final String clazz, final Class<?> callingClass) throws ClassNotFoundException
    {
        return BundleClassLoaderAccessor.loadClass(getBundleOrFail(), clazz);
    }

    public URL getResource(final String name)
    {
        return getBundleClassLoaderOrFail().getResource(name);
    }

    public InputStream getResourceAsStream(final String name)
    {
        return getBundleClassLoaderOrFail().getResourceAsStream(name);
    }

    @Override
    protected void installInternal() throws OsgiContainerException, IllegalPluginStateException
    {
        super.installInternal();
        if (null != osgiContainerManager)
        {
            // We're pending installation, so install
            final File file = pluginArtifact.toFile();
            final boolean allowReference = PluginArtifact.AllowsReference.Default.allowsReference(pluginArtifact);
            bundle = OsgiContainerManager.AllowsReferenceInstall.Default.installBundle(osgiContainerManager, file, allowReference);
            bundleClassLoader = BundleClassLoaderAccessor.getClassLoader(bundle, new AlternativeDirectoryResourceLoader());
            osgiContainerManager = null;
        }
        else if (null == bundle)
        {
            throw new IllegalPluginStateException("Cannot reuse instance for bundle '" + getKey() + "'");
        }
        // else this could be a reinstall or not, but we can't tell, so we let it slide
    }

    @Override
    protected void uninstallInternal()
    {
        try
        {
            if (bundleIsUsable("uninstall"))
            {
                if (bundle.getState() != Bundle.UNINSTALLED)
                {
                    bundle.uninstall();
                }
                else
                {
                    // A previous uninstall aborted early ?
                    log.warn("Bundle '{}' already UNINSTALLED, but still held", getKey());
                }
                bundle = null;
                bundleClassLoader = null;
            }
        }
        catch (final BundleException e)
        {
            throw new PluginException(e);
        }
    }

    @Override
    protected PluginState enableInternal()
    {
        log.debug("Enabling OSGi bundled plugin '{}'", getKey());
        try
        {
            if (bundleIsUsable("enable"))
            {
                if (bundle.getHeaders().get(Constants.FRAGMENT_HOST) == null)
                {
                    log.debug("Plugin '{}' bundle is NOT a fragment, starting.", getKey());
                    bundle.start();
                    bundle.getBundleContext().addBundleListener(bundleStartStopListener);
                }
                else
                {
                    log.debug("Plugin '{}' bundle is a fragment, not doing anything.", getKey());
                }
            }

            return PluginState.ENABLED;
        }
        catch (final BundleException e)
        {
            throw new PluginException(e);
        }
    }

    @Override
    protected void disableInternal()
    {
        try
        {
            if (bundleIsUsable("disable"))
            {
                if (bundle.getState() == Bundle.ACTIVE)
                {
                    bundle.stop();
                }
                else
                {
                    log.warn("Cannot disable Bundle '{}', not ACTIVE", getKey());
                }
            }
        }
        catch (final BundleException e)
        {
            throw new PluginException(e);
        }
    }

    public ClassLoader getClassLoader()
    {
        return getBundleClassLoaderOrFail();
    }

    public PluginArtifact getPluginArtifact()
    {
        return pluginArtifact;
    }

    private String getInstallationStateExplanation()
    {
        return (null != osgiContainerManager) ? "not yet installed" : "already uninstalled";
    }

    private boolean bundleIsUsable(final String task)
    {
        if (null != bundle)
        {
            return true;
        }
        else
        {
            final String why = getInstallationStateExplanation();
            log.warn("Cannot {} {} bundle '{}'", new Object[] { task, why, getKey()});
            return false;
        }
    }

    private <T> T getOrFail(final T what, final String name) throws PluginException
    {
        if (null == what)
        {
            throw new IllegalPluginStateException("Cannot use " + name + " of " + getInstallationStateExplanation() + " '"
                    + getKey() + "' from '" + pluginArtifact + "'");
        }
        else
        {
            return what;
        }
    }

    private Bundle getBundleOrFail() throws PluginException
    {
        return getOrFail(bundle, "bundle");
    }

    private ClassLoader getBundleClassLoaderOrFail() throws PluginException
    {
        return getOrFail(bundleClassLoader, "bundleClassLoader");
    }
}
