package com.atlassian.maven.plugins.sandbox;

import com.atlassian.sandbox.model.SandboxRepository;
import com.atlassian.sandbox.model.SandboxResult;
import com.atlassian.sandbox.model.VCSFingerprint;
import com.google.gson.Gson;
import org.apache.commons.io.FileUtils;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.repository.ArtifactRepositoryFactory;
import org.apache.maven.artifact.repository.layout.DefaultRepositoryLayout;
import org.apache.maven.model.DeploymentRepository;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.util.StringUtils;
import org.tmatesoft.svn.core.SVNException;
import org.tmatesoft.svn.core.wc.SVNClientManager;
import org.tmatesoft.svn.core.wc.SVNInfo;
import org.tmatesoft.svn.core.wc.SVNRevision;
import org.tmatesoft.svn.core.wc.SVNStatus;

import java.io.File;
import java.io.IOException;

/**
 * Modifies the distribution management of the project so that all builds dep
 *
 * @phase verify
 * @goal sandbox
 */
public class SandboxMojo extends AbstractMojo
{
    private static final Gson GSON = new Gson();

    /**
     * The Maven Project
     *
     * @parameter expression="${project}"
     * @required
     * @readonly
     */
    private MavenProject project;

    /**
     * URL to the sandbox repository server
     * 
     * @parameter expression="${sandbox.repositoryUrl}" default-value="dav:http://localhost/"
     */
    private String sandboxRepositoryUrl;

    /**
     * @parameter expression="${sandbox.chainResultKey}"
     */
    private String chainResultKey;

    /**
     * @parameter expression="${sandbox.baseRepositoryUrl}"
     */
    private String repositoryBaseUrl;

    public MavenProject getProject()
    {
        return project;
    }

    public String getSandboxRepositoryRootUrl()
    {
        if (!StringUtils.isEmpty(sandboxRepositoryUrl))
            return StringUtils.strip(sandboxRepositoryUrl);
        return sandboxRepositoryUrl;
    }

    private static final String SANDBOX_RESULT_FILE_NAME = "sandbox.json";

    /**
     * @component
     */
    private ArtifactRepositoryFactory repositoryFactory;

    /**
     * The output directory of the project.
     *
     * @parameter default-value="${project.build.directory}" expression="${sandbox.outputDirectory}"
     */
    protected String outputDirectory;

    public void execute()
        throws MojoExecutionException, MojoFailureException
    {
        if (isEnabled())
        {
            getLog().info("Sandbox enabled");
            setArtifactRepositories();

            SandboxResult sandboxResult = createResult();
            writeResult(sandboxResult);
        }
    }

    private boolean isEnabled()
    {
        return StringUtils.isNotEmpty(sandboxRepositoryUrl) && StringUtils.isNotEmpty(chainResultKey) && StringUtils.isNotEmpty(repositoryBaseUrl);
    }

    private SandboxResult createResult() throws MojoExecutionException
    {
        SandboxResult sandboxResult = new SandboxResult();
        final ArtifactRepository artifactRepository = getProject().getDistributionManagementArtifactRepository();
        SandboxRepository sandboxRepository = new SandboxRepository(artifactRepository.getId(), artifactRepository.getUrl());
        sandboxResult.addRepository(sandboxRepository);
        return sandboxResult;
    }

    private void writeResult(final SandboxResult sandboxResult) throws MojoExecutionException
    {
        final File sandboxFile = new File(outputDirectory, SANDBOX_RESULT_FILE_NAME);
        if (!sandboxFile.getParentFile().exists())
        {
            sandboxFile.getParentFile().mkdirs();
        }

        try
        {
            FileUtils.writeStringToFile(sandboxFile, GSON.toJson(sandboxResult));
        }
        catch (IOException e)
        {
            throw new MojoExecutionException("Could not write sandbox result for this module to '" + sandboxFile.getAbsolutePath() + "'", e);
        }
    }

    private void setArtifactRepositories() throws MojoExecutionException
    {
        getLog().info("Sandboxing deployment repositories to " + getRepositorySandboxUrl());
        //Set release
        ArtifactRepository releaseRepository = createDeployableArtifactRepository(getProject().getDistributionManagement().getRepository());
        getProject().setReleaseArtifactRepository(releaseRepository);

        //Set snapshot
        ArtifactRepository snapshotRepository = createDeployableArtifactRepository(getProject().getDistributionManagement().getSnapshotRepository());
        getProject().setSnapshotArtifactRepository(snapshotRepository);
    }

    private ArtifactRepository createDeployableArtifactRepository(DeploymentRepository deploymentRepository) throws MojoExecutionException
    {
        final String newDeploymentUrl = getRepositorySandboxUrl() + "/" + getRepositoryName(deploymentRepository.getUrl());
        getLog().info("Repository " + deploymentRepository.getUrl() + " has been sandboxed to " + newDeploymentUrl);
        return repositoryFactory.createDeploymentArtifactRepository(deploymentRepository.getId(), newDeploymentUrl, new DefaultRepositoryLayout(), deploymentRepository.isUniqueVersion());
    }

    private String getRepositorySandboxUrl()
    {
        String url = getSandboxRepositoryRootUrl();
        if (!url.endsWith("/"))
            url += "/";

        return url + chainResultKey + "/";
    }

    public String getRepositoryName(String url)
    {
        String[] parts = StringUtils.split(url, "/");
        if (parts.length > 0)
        {
            return parts[parts.length - 1];
        }
        return null;
    }
}
