package com.atlassian.event.internal;

import com.atlassian.event.api.EventPublisher;
import com.atlassian.event.config.ListenerHandlersConfiguration;
import com.atlassian.event.spi.EventDispatcher;
import com.atlassian.event.spi.ListenerHandler;
import com.atlassian.event.spi.ListenerInvoker;
import com.atlassian.plugin.scope.EverythingIsActiveScopeManager;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.MapMaker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;

import static com.atlassian.event.internal.EventPublisherUtils.getInvokersWithClassHierarchyOrder;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A non-blocking implementation of the {@link com.atlassian.event.api.EventPublisher} interface.
 * <p>
 * This class is a drop-in replacement for {@link EventPublisherImpl} except that it does not
 * synchronise on the internal map of event type to {@link ListenerInvoker}, and should handle
 * much higher parallelism of event dispatch.
 * <p>
 * One can customise the event listening by instantiating with custom {@link ListenerHandler listener handlers} and
 * the event dispatching through {@link EventDispatcher}. See the {@link com.atlassian.event.spi} package for more
 * information.
 *
 * @see ListenerHandler
 * @see EventDispatcher
 * @since 2.0.2
 */
public final class LockFreeEventPublisher implements EventPublisher {
    private static final EverythingIsActiveScopeManager SCOPE_MANAGER = new EverythingIsActiveScopeManager();
    /**
     * Gets the {@link ListenerInvoker invokers} for a listener
     */
    private final InvokerBuilder invokerBuilder;

    /**
     * Publishes an event.
     */
    private final Publisher publisher;

    /**
     * <strong>Note:</strong> this field makes this implementation stateful
     */
    private final Listeners listeners = new Listeners();

    public LockFreeEventPublisher(final EventDispatcher eventDispatcher,
                                  final ListenerHandlersConfiguration listenerHandlersConfiguration) {
        this(eventDispatcher, listenerHandlersConfiguration, (invokers, event) -> invokers);
    }

    /**
     * If you need to customise the asynchronous handling, you should use the
     * {@link com.atlassian.event.internal.AsynchronousAbleEventDispatcher}
     * together with a custom executor.
     * <p>
     * You might also want to have a look at using the
     * {@link com.atlassian.event.internal.EventThreadFactory} to keep the naming
     * of event threads consistent with the default naming of the Atlassian Event
     * library.
     *
     * @param eventDispatcher               the event dispatcher to be used with the publisher
     * @param listenerHandlersConfiguration the list of listener handlers to be used with this publisher
     * @param transformer                   the batcher for batching up listener invocations
     * @see com.atlassian.event.internal.AsynchronousAbleEventDispatcher
     * @see com.atlassian.event.internal.EventThreadFactory
     */

    public LockFreeEventPublisher(final EventDispatcher eventDispatcher,
                                  final ListenerHandlersConfiguration listenerHandlersConfiguration,
                                  final InvokerTransformer transformer) {
        invokerBuilder = new InvokerBuilder(checkNotNull(listenerHandlersConfiguration).getListenerHandlers());
        publisher = new Publisher(eventDispatcher, listeners, transformer);
    }

    public void publish(final @Nonnull Object event) {
        checkNotNull(event);
        publisher.dispatch(event);
    }

    public void register(final @Nonnull Object listener) {
        checkNotNull(listener);
        listeners.register(listener, invokerBuilder.build(listener));
    }

    public void unregister(final @Nonnull Object listener) {
        checkNotNull(listener);
        listeners.remove(listener);
    }

    public void unregisterAll() {
        listeners.clear();
    }

    //
    // inner classes
    //

    /**
     * Maps classes to the relevant {@link Invokers}
     */
    static final class Listeners {
        /**
         * We always want an {@link Invokers} created for any class requested, even if it is empty.
         * <b>Warning:</b> We need to use weakKeys here to ensure plugin event classes are GC'd, otherwise we leak...
         */
        private final LoadingCache<Class<?>, Invokers> invokers = CacheBuilder.newBuilder().build(new CacheLoader<Class<?>, Invokers>() {
            @Override
            public Invokers load(final @Nonnull Class<?> key) {
                return new Invokers();
            }
        });

        void register(final Object listener, final Iterable<ListenerInvoker> invokers) {
            for (final ListenerInvoker invoker : invokers) {
                register(listener, invoker);
            }
        }

        private void register(final Object listener, final ListenerInvoker invoker) {
            Set<Class<?>> supportedEventTypes = invoker.getSupportedEventTypes();
            if (supportedEventTypes.isEmpty()) {
                // if supported classes is empty, then all events are supported.
                supportedEventTypes = Collections.singleton(Object.class);
            }

            for (final Class<?> eventClass : supportedEventTypes) {
                invokers.getUnchecked(eventClass).add(listener, new ListenerInvokerWithRegisterOrder(listener, invoker, Optional.empty()));
            }
        }

        void remove(final Object listener) {
            invokers.asMap().forEach((k, listeners) -> listeners.remove(listener));
        }

        void clear() {
            invokers.invalidateAll();
        }

        public Collection<ListenerInvokerWithRegisterOrder> get(final Class<?> eventClass) {
            return invokers.getUnchecked(eventClass).all();
        }
    }

    /**
     * map of Key to Set of ListenerInvoker
     */
    static final class Invokers {
        private final ConcurrentMap<Object, Collection<ListenerInvokerWithRegisterOrder>> listeners = new MapMaker().weakKeys().makeMap();

        Collection<ListenerInvokerWithRegisterOrder> all() {
            return listeners.values().stream()
                    .flatMap(Collection::stream)
                    .collect(Collectors.toList());
        }

        void remove(final Object key) {
            listeners.remove(key);
        }

        void add(final Object key, final ListenerInvokerWithRegisterOrder invoker) {
            listeners.computeIfAbsent(key, k -> new ArrayList<>()).add(invoker);
        }
    }

    /**
     * Responsible for publishing an event.
     * <p>
     * Must first get the Set of all ListenerInvokers that
     * are registered for that event and then use the
     * {@link EventDispatcher} to send the event to them.
     */
    static final class Publisher {
        private final Logger log = LoggerFactory.getLogger(this.getClass());
        private final Listeners listeners;
        private final EventDispatcher dispatcher;
        private final InvokerTransformer transformer;

        Publisher(final EventDispatcher dispatcher, final Listeners listeners, final InvokerTransformer transformer) {
            this.dispatcher = checkNotNull(dispatcher);
            this.listeners = checkNotNull(listeners);
            this.transformer = checkNotNull(transformer);
        }

        void dispatch(final Object event) {
            Iterable<ListenerInvoker> invokers = findListenerInvokersForEvent(event);
            try {
                invokers = transformer.transformAll(invokers, event);
            } catch (Exception e) {
                log.error("Exception while transforming invokers. Dispatching original invokers instead.", e);
            }
            for (final ListenerInvoker invoker : invokers) {
                // EVENT-14 -  we should continue to process all listeners even if one throws some horrible exception
                try {
                    dispatcher.dispatch(invoker, event);
                } catch (Exception e) {
                    log.error("There was an exception thrown trying to dispatch event '{}' from the invoker '{}'.",
                            event, invoker, e);
                }
            }
        }

        private Set<ListenerInvoker> findListenerInvokersForEvent(final Object event) {
            final Set<ListenerInvokerWithClassHierarchyAndRegisterOrder> invokers = getInvokersWithClassHierarchyOrder(event, listeners::get);
            return EventPublisherUtils.sortInvokers(SCOPE_MANAGER, invokers);
        }
    }

    /**
     * Holds all configured {@link ListenerHandler handlers}
     */
    static final class InvokerBuilder {
        private final Iterable<ListenerHandler> listenerHandlers;

        InvokerBuilder(final @Nonnull Iterable<ListenerHandler> listenerHandlers) {
            this.listenerHandlers = checkNotNull(listenerHandlers);
        }

        Iterable<ListenerInvoker> build(final Object listenerOrMd) throws IllegalArgumentException {
            final Object listener = EventPublisherUtils.getListener(listenerOrMd);
            final ImmutableList.Builder<ListenerInvoker> builder = ImmutableList.builder();
            for (final ListenerHandler listenerHandler : listenerHandlers) {
                builder.addAll(listenerHandler.getInvokers(listener));
            }
            final List<ListenerInvoker> invokers = builder.build();
            if (invokers.isEmpty()) {
                throw new IllegalArgumentException("No listener invokers were found for listener <" + listener + ">");
            }
            return invokers;
        }
    }
}
