package com.atlassian.event.internal;

import com.atlassian.event.api.EventListener;
import com.atlassian.event.spi.ListenerInvoker;
import com.atlassian.plugin.eventlistener.descriptors.EventListenerModuleDescriptor;
import com.atlassian.plugin.scope.ScopeManager;

import java.util.Collection;
import java.util.Comparator;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.function.ToIntFunction;
import java.util.stream.Collectors;

import static com.google.common.base.Preconditions.checkNotNull;

public class EventPublisherUtils {
    public static Set<ListenerInvokerWithClassHierarchyAndRegisterOrder> getInvokersWithClassHierarchyOrder(
            final Object event,
            final Function<Class<?>, Collection<ListenerInvokerWithRegisterOrder>> eventToListeners) {
        final Set<ListenerInvokerWithClassHierarchyAndRegisterOrder> invokers = new LinkedHashSet<>();
        final AtomicInteger classHierarchyOrder = new AtomicInteger();
        for (final Class<?> eventClass : ClassUtils.findAllTypes(checkNotNull(event).getClass())) {
            invokers.addAll(eventToListeners.apply(eventClass).stream()
                    .map(invoker -> new ListenerInvokerWithClassHierarchyAndRegisterOrder(invoker, classHierarchyOrder.get()))
                    .collect(Collectors.toList()));
            classHierarchyOrder.incrementAndGet();
        }
        return invokers;
    }

    /**
     * Use following properties for sorting:
     * <ol>
     *     <li>{@link EventListener#order()} </li>
     *     <li>event class hierarchy (from bottom to top)</li>
     *     <li>event listeners registering order</li>
     * </ol>
     */
    static Set<ListenerInvoker> sortInvokers(final ScopeManager scopeManager, final Set<ListenerInvokerWithClassHierarchyAndRegisterOrder> invokers) {
        return invokers.stream()
                .filter(i -> i.keyedListenerInvoker.getScope()
                        .map(scopeManager::isScopeActive)
                        .orElse(true))
                .sorted(
                        Comparator
                                .comparingInt((ToIntFunction<ListenerInvokerWithClassHierarchyAndRegisterOrder>) value -> value.getListenerInvokerWithRegisterOrder().getOrder())
                                .thenComparingInt(value -> value.classHierarchyOrder)
                                .thenComparingInt(value -> value.getListenerInvokerWithRegisterOrder().getRegisterOrder()))
                .map(ListenerInvokerWithClassHierarchyAndRegisterOrder::getListenerInvokerWithRegisterOrder)
                .map(ListenerInvokerWithRegisterOrder::getInvoker)
                .collect(Collectors.toCollection(LinkedHashSet::new));
    }

    static Object getListener(final Object listener) {
        if (listener instanceof EventListenerModuleDescriptor) {
            final EventListenerModuleDescriptor descriptor = (EventListenerModuleDescriptor) listener;
            return descriptor.getModule();
        } else {
            return listener;
        }
    }
}
