package com.atlassian.event.internal;

import com.atlassian.event.api.EventPublisher;
import com.atlassian.event.config.ListenerHandlersConfiguration;
import com.atlassian.event.spi.EventDispatcher;
import com.atlassian.event.spi.ListenerHandler;
import com.atlassian.event.spi.ListenerInvoker;
import com.atlassian.plugin.eventlistener.descriptors.EventListenerModuleDescriptor;
import com.atlassian.plugin.scope.EverythingIsActiveScopeManager;
import com.atlassian.plugin.scope.ScopeManager;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import static com.atlassian.event.internal.EventPublisherUtils.getInvokersWithClassHierarchyOrder;
import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.commons.lang3.ObjectUtils.identityToString;

/**
 * The default implementation of the {@link com.atlassian.event.api.EventPublisher} interface.
 * <p>
 * One can customise the event listening by instantiating with custom {@link ListenerHandler listener handlers} and
 * the event dispatching through {@link EventDispatcher}.
 * <p>
 * See the {@link com.atlassian.event.spi} package for more information.
 *
 * @see ListenerHandler
 * @see EventDispatcher
 * @since 2.0
 */
public final class EventPublisherImpl implements EventPublisher {
    private static final Logger log = LoggerFactory.getLogger(EventPublisherImpl.class);

    private static final String PROPERTY_PREFIX = EventPublisherImpl.class.getName();
    private static final Optional<String> debugRegistration =
            Optional.ofNullable(System.getProperty(PROPERTY_PREFIX + ".debugRegistration"));
    private static final boolean debugRegistrationLocation =
            Boolean.getBoolean(PROPERTY_PREFIX + ".debugRegistrationLocation");
    private static final Optional<String> debugInvocation =
            Optional.ofNullable(System.getProperty(PROPERTY_PREFIX + ".debugInvocation"));
    private static final boolean debugInvocationLocation =
            Boolean.getBoolean(PROPERTY_PREFIX + ".debugInvocationLocation");

    private final EventDispatcher eventDispatcher;
    private final List<ListenerHandler> listenerHandlers;
    private final ScopeManager scopeManager;

    /**
     * <strong>Note:</strong> this field makes this implementation stateful
     */
    private final Multimap<Class<?>, ListenerInvokerWithRegisterOrder> listenerInvokers;

    /**
     * If you need to customise the asynchronous handling, you should use the {@link AsynchronousAbleEventDispatcher}
     * together with a custom executor. You might also want to have a look at using the {@link EventThreadFactory} to
     * keep the naming of event threads consistent with the default naming of the Atlassian Event library.
     *
     * @param eventDispatcher               the event dispatcher to be used with the publisher
     * @param listenerHandlersConfiguration the list of listener handlers to be used with this publisher
     * @see AsynchronousAbleEventDispatcher
     * @see EventThreadFactory
     */
    public EventPublisherImpl(EventDispatcher eventDispatcher, ListenerHandlersConfiguration listenerHandlersConfiguration) {
        this(eventDispatcher, listenerHandlersConfiguration, new EverythingIsActiveScopeManager());
    }

    /**
     * Inherits {@link EventPublisherImpl#EventPublisherImpl(EventDispatcher, ListenerHandlersConfiguration)} and
     * allows injection of scope manager
     *
     * @param eventDispatcher               the event dispatcher to be used with the publisher
     * @param listenerHandlersConfiguration the list of listener handlers to be used with this publisher
     * @param scopeManager                  the scope manager
     * @see AsynchronousAbleEventDispatcher
     * @see EventThreadFactory
     * @see ScopeManager
     */
    public EventPublisherImpl(EventDispatcher eventDispatcher,
                              ListenerHandlersConfiguration listenerHandlersConfiguration,
                              ScopeManager scopeManager) {
        this.eventDispatcher = checkNotNull(eventDispatcher);
        this.listenerHandlers = checkNotNull(checkNotNull(listenerHandlersConfiguration).getListenerHandlers());
        this.listenerInvokers = newMultimap();
        this.scopeManager = checkNotNull(scopeManager);
    }

    /**
     * Check {@link EventPublisher#publish(Object)} for details about the order of processing events.
     *
     * @param event the event to publish
     */
    public void publish(@Nonnull Object event) {
        invokeListeners(findListenerInvokersForEvent(checkNotNull(event)), event);
    }

    public void register(@Nonnull final Object listener) {
        registerListener(listener);
    }

    public void unregister(@Nonnull Object listener) {
        unregisterListener(listener);
    }

    public void unregisterAll() {
        synchronized (listenerInvokers) {
            listenerInvokers.clear();
        }
    }

    private void unregisterListener(Object listener) {
        // see Multimaps#synchronizedMultimap(Multimap) for why this synchronize block is there
        synchronized (listenerInvokers) {
            listenerInvokers.entries().removeIf(entry -> entry.getValue().isFor(listener));
        }
    }

    private void registerListener(Object listener) {
        final Optional<String> parentScope;
        if (listener instanceof EventListenerModuleDescriptor) {
            final EventListenerModuleDescriptor descriptor = (EventListenerModuleDescriptor) listener;
            parentScope = descriptor.getScopeKey();
        } else {
            parentScope = Optional.empty();
        }

        final Object listenerImpl = EventPublisherUtils.getListener(listener);
        synchronized (listenerInvokers) /* Because we need to un-register an re-register in one 'atomic' operation */ {
            unregisterListener(listener);

            final List<ListenerInvoker> invokers = Lists.newArrayList();
            for (ListenerHandler listenerHandler : listenerHandlers) {
                invokers.addAll(listenerHandler.getInvokers(listenerImpl));
            }
            if (!invokers.isEmpty()) {
                registerListenerInvokers(listener, parentScope, invokers);
            } else {
                throw new IllegalArgumentException("No listener invokers were found for listener <" + listenerImpl + ">");
            }
        }
    }

    private Set<ListenerInvoker> findListenerInvokersForEvent(final Object event) {
        final Set<ListenerInvokerWithClassHierarchyAndRegisterOrder> invokers;
        // see Multimaps#synchronizedMultimap(Multimap) for why this synchronize block is there */
        synchronized (listenerInvokers) {
            invokers = getInvokersWithClassHierarchyOrder(event, listenerInvokers::get);
        }
        return EventPublisherUtils.sortInvokers(scopeManager, invokers);
    }

    private void invokeListeners(Collection<ListenerInvoker> listenerInvokers, Object event) {
        final String eventClass = event.getClass().getName();
        final boolean debugThisInvocation = debugInvocation.map(eventClass::startsWith).orElse(false);
        for (final ListenerInvoker invoker : listenerInvokers) {
            if (debugThisInvocation) {
                log.warn("Listener invoked event with class '{}' -> invoker {}", eventClass, invoker);
                if (debugInvocationLocation) {
                    log.warn("Invoked from", new Exception());
                }
            }
            // EVENT-14 -  we should continue to process all listeners even if one throws some horrible exception
            try {
                eventDispatcher.dispatch(invoker, event);
            } catch (Exception e) {
                log.error("There was an exception thrown trying to dispatch event '{}' from the invoker '{}'.",
                        event, invoker, e);
            }
        }
    }

    private void registerListenerInvokers(Object listener, Optional<String> parentScope,
                                          List<? extends ListenerInvoker> invokers) {
        for (ListenerInvoker invoker : invokers) {
            registerListenerInvoker(listener, parentScope, invoker);
        }
    }

    private void registerListenerInvoker(Object listener, Optional<String> parentScope, ListenerInvoker invoker) {
        // if supported classes is empty, then all events are supported.
        if (invoker.getSupportedEventTypes().isEmpty()) {
            putEventListenerInvoker(Object.class, listener, parentScope, invoker);
        }

        // if it it empty, we won't loop, otherwise register the invoker against all its classes
        for (Class<?> eventClass : invoker.getSupportedEventTypes()) {
            putEventListenerInvoker(eventClass, listener, parentScope, invoker);
        }
    }

    private void putEventListenerInvoker(final Class<?> eventClass, Object listener, Optional<String> parentScope,
                                         ListenerInvoker invoker) {
        debugRegistration.ifPresent(classPrefix -> {
            if (eventClass.getName().startsWith(classPrefix)) {
                log.warn("Listener registered event '{}' -> invoker {}", eventClass, invoker);
                if (debugRegistrationLocation) {
                    log.warn("Registered from", new Exception());
                }
            }
        });

        final String listenerKey = identityToString(listener);
        log.debug("Registering {} with scope {} and order {}", listenerKey, parentScope, invoker.getOrder());

        if(parentScope.isPresent() && invoker.getScope().isPresent() && !parentScope.equals(invoker.getScope())) {
            throw new IllegalArgumentException("Listener <" + listenerKey + "> tries to override parent scope <" + parentScope.get() + "> with <" + invoker.getScope() + ">");
        }

        final Optional<String> invokerScope = parentScope.isPresent()
                ?parentScope
                :invoker.getScope();
        final int order = invoker.getOrder();

        listenerInvokers.put(eventClass, new ListenerInvokerWithRegisterOrder(listener, invoker, invokerScope));
    }

    private Multimap<Class<?>, ListenerInvokerWithRegisterOrder> newMultimap() {
        return Multimaps.synchronizedMultimap(
                Multimaps.newMultimap(new HashMap<>(), HashSet::new));
    }
}
