package com.atlassian.confluence.rest.client;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.content.Content;
import com.atlassian.confluence.api.model.content.ContentBody;
import com.atlassian.confluence.api.model.content.ContentStatus;
import com.atlassian.confluence.api.model.content.id.ContentId;
import com.atlassian.confluence.api.model.content.ContentRepresentation;
import com.atlassian.confluence.api.model.content.ContentType;
import com.atlassian.confluence.api.model.content.Space;
import com.atlassian.confluence.api.model.locator.ContentLocator;
import com.atlassian.confluence.api.model.pagination.PageRequest;
import com.atlassian.confluence.api.model.pagination.PageResponse;
import com.atlassian.confluence.api.service.content.ContentBodyConversionService;
import com.atlassian.confluence.api.service.content.ContentService;
import com.atlassian.confluence.api.service.exceptions.ServiceException;
import com.atlassian.fugue.Option;
import com.atlassian.util.concurrent.Promise;
import org.joda.time.LocalDate;

import java.util.Map;

/**
 * {@link ContentService} implementation that communicates with Confluence remotely using the Confluence REST API.
 *
 * Provides future returning equivalents for the methods in ContentService.
 */
@ExperimentalApi
public interface RemoteContentService
{
    /**
     * Create a content finder to retrieve content from the remote server
     *
     * @return the content, if one exists matching the id, or none otherwise.
     */
    RemoteContentFinder find(Expansion... expansions);

    interface RemoteContentFinder extends RemoteParameterContentFinder
    {
        RemoteSingleContentFetcher withId(ContentId contentId);

        /**
         * Retrieve a piece of content by some more complex criteria (defined by the ContentLocator).
         *
         * @param locator the locator containing the criteria for the content you are looking for
         * @return the content, if one exists matching the criteria, or none otherwise.
         */
        RemoteSingleContentFetcher withLocator(ContentLocator locator);

        RemoteContentFinder withStatus(ContentStatus... status);

        RemoteContentFinder withAnyStatus();
    }

    interface RemoteParameterContentFinder extends RemoteContentFetcher
    {
        RemoteParameterContentFinder withSpace(Space... space);

        RemoteParameterContentFinder withType(ContentType... type);

        RemoteParameterContentFinder withCreatedDate(LocalDate time);

        RemoteParameterContentFinder withTitle(String title);
    }

    interface RemoteSingleContentFetcher
    {
        Promise<Option<Content>> fetchOne();

        Promise<Content> fetchOneOrNull();
    }

    interface RemoteContentFetcher extends RemoteSingleContentFetcher
    {
        Promise<PageResponse<Content>> fetchMany(ContentType type, PageRequest request);

        Promise<Map<ContentType, PageResponse<Content>>> fetchMappedByContentType(PageRequest request);
    }

    /**
     * Create a piece of content.
     * <p>
     * Valid {@link ContentRepresentation}s for the {@link ContentBody} can be found in the documentation on the
     * {@link ContentBodyConversionService}.
     * </p>
     *
     * @param newContent the content to create
     * @return the content created
     * @throws ServiceException if the content cannot be created
     */
    Promise<Content> create(Content newContent);

    /**
     * Updates a piece of content.
     *
     * <p>
     * Valid {@link ContentRepresentation}s for the {@link ContentBody} can be found in the documentation on the
     * {@link ContentBodyConversionService}.
     * </p>
     *
     * @param content the updated {@link Content} with metadata about the change
     * @return the updated content after being persisted
     */
    Promise<Content> update(Content content);

    /**
     * Trash a piece of content.
     *
     * <p>
     *     The {@link #delete(Content)} method should be used for content that doesn't support trashing.
     *     Currently Pages and Blogposts support trashing.
     * </p>
     *
     * @param content the content to be trashed
     * @return the content after being trashed
     */
    Promise<Content> trash(Content content);

    /**
     * Restore a trashed piece of content.
     *
     * @param content the content to be restored
     * @return the content after being restored
     */
    Promise<Content> restore(Content content);

    /**
     * Permanently delete a trashed piece of content.
     *
     * @param content the content to be purged
     */
    Promise<Void> purge(Content content);

    /**
     * Removes an item of Content from the system.
     *
     *
     * @param content if of the content to remove
     * @throws ServiceException if the content cannot be found, or cannot be deleted
     */
    Promise<Void> delete(Content content);

    /**
     * Retrieve the children of an item of Content.
     *
     * <p>
     *     Children will be {@link Content} items of the same type as the parent.
     * </p>
     * <p>
     * The pageRequest limit is restricted to a maximum page size.
     *
     * @param parent - the content to retrieve the children for
     * @param pageRequest - a pageRequest indicating how much content to retrieve.
     * @return a partial list of the top level content in this space.
     */
    Promise<PageResponse<Content>> getChildren(Content parent, PageRequest pageRequest, Expansion... expansion);
}
