package com.atlassian.confluence.rest.client;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.confluence.api.model.content.id.ContentId;
import com.atlassian.confluence.api.model.content.Label;
import com.atlassian.confluence.api.model.pagination.PageRequest;
import com.atlassian.confluence.api.service.exceptions.NotFoundException;
import com.atlassian.confluence.api.service.exceptions.ServiceException;
import com.atlassian.util.concurrent.Promise;

import java.util.Collection;

/**
 * {@link com.atlassian.confluence.api.service.content.ContentLabelService} implementation that communicates with Confluence
 * remotely using the Confluence REST API.
 *
 * Provides {@link Promise} returning equivalents for the methods in ContentLabelService.
 */
@ExperimentalApi
public interface RemoteContentLabelService
{
    /**
     * Get the labels attached to a given piece of content in the given namespaces
     *
     * @param contentId the id of the content to retrieve
     * @throws com.atlassian.confluence.api.service.exceptions.NotFoundException if the content does not exist, or is not viewable by the user
     */
    Promise<? extends Iterable<Label>> getLabels(ContentId contentId, Collection<Label.Prefix> prefixes, PageRequest request) throws NotFoundException;

    /**
     * Adds the given labels to the specified content
     *
     * @param contentId the id of the content to add labels to
     * @param labels the label(s) that will be added
     * @return all the labels that are associated with the specified content
     * @throws com.atlassian.confluence.api.service.exceptions.ServiceException if at least one label is invalid. Provides an error message
     */
    Promise<? extends Iterable<Label>> addLabels(ContentId contentId, Iterable<Label> labels) throws ServiceException;

    /**
     * Remove a label with labelId from the specified content
     * @param contentId the id of the content to remove the label from
     * @param labels the labels to remove
     * @throws ServiceException failed to remove the label
     */
    void removeLabels(ContentId contentId, Iterable<Label> labels) throws ServiceException;
}
