package com.atlassian.confluence.rest.client;

import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.pagination.PageRequest;
import com.atlassian.confluence.api.model.pagination.PageResponse;
import com.atlassian.confluence.api.service.exceptions.ServiceException;
import com.atlassian.confluence.rest.api.model.ExpansionsParser;
import com.atlassian.confluence.rest.api.model.RestList;
import com.atlassian.confluence.rest.client.authentication.AuthenticatedWebResourceProvider;
import com.atlassian.fugue.Option;
import com.atlassian.util.concurrent.Promise;

import com.google.common.base.Predicate;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.GenericType;
import com.sun.jersey.api.client.UniformInterfaceException;
import com.sun.jersey.api.client.WebResource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import javax.ws.rs.core.MediaType;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Map;
import java.util.concurrent.Callable;

import static com.atlassian.confluence.rest.api.model.ExceptionConverter.Client.convertToServiceException;

/**
 * Common parent class of all Confluence API Remote Service implementations, handling authentication and REST
 * infrastructure.
 *
 * @param <P> - the interface of the local service that this is a remote service of, used by RemoteServiceProxyCreator
 */
@SuppressWarnings("UnusedDeclaration")
public abstract class AbstractRemoteService<P>
{
    private static final Logger log = LoggerFactory.getLogger(AbstractRemoteService.class);

    private AuthenticatedWebResourceProvider provider;

    private final PromisingExecutorService executor;

    protected AbstractRemoteService(AuthenticatedWebResourceProvider provider, ListeningExecutorService executor)
    {
        this.provider = provider;
        this.executor = new PromisingExecutorService(executor);
    }

    protected AbstractRemoteService(AbstractRemoteService other)
    {
        provider = other.provider;
        executor = other.executor;
    }

    protected WebResource newRestWebResource()
    {
        return provider.newRestWebResource().path("rest").path("api");
    }

    protected WebResource newExperimentalRestWebResource()
    {
        return provider.newRestWebResource().path("rest").path("experimental");
    }

    protected <T> Promise<Option<T>> getFutureOption(final WebResource resource, final Class<? extends T> entityClass)
    {
        return executor.submit(new Callable<Option<T>>()
        {
            @Override
            public Option<T> call() throws Exception
            {
                return getOption(resource, entityClass);
            }
        });
    }

    protected <E> Promise<PageResponse<E>> getFuturePageResponseList(final WebResource resource, final Class<E> contentClass)
    {
        return executor.submit(new Callable<PageResponse<E>>()
        {
            @Override
            public PageResponse<E> call() throws Exception
            {
                return getPartialList(resource, contentClass);
            }
        });
    }

    protected <K,V> Promise<Map<K, PageResponse<V>>> getFutureMapOfPageResponses(final WebResource resource, final Class<K> keyClass, final Class<V> listContentClass)
    {
        final GenericType<Map> mapType = getGenericType(Map.class, keyClass, getParameterizedType(RestList.class, listContentClass));
        Promise promise = getFutureGenericMap(resource, mapType);
        return promise;

    }

    protected <K,V> Promise<Map<K, V>> getFutureMap(final WebResource resource, final Class<K> keyClass, final Class<V> valueClass)
    {
        final GenericType<Map> mapType = getGenericType(Map.class, keyClass, valueClass);
        Promise promise = getFutureGenericMap(resource, mapType);
        return promise;
    }

    private <K,V> Promise<Map> getFutureGenericMap(final WebResource resource, final GenericType<Map> mapType)
    {
        return executor.submit(new Callable<Map>()
        {
            @Override
            public Map call() throws Exception
            {
                try
                {
                    // FIXME need to not deserialize _links as an entry
                    return Maps.filterEntries(resource.get(mapType), new Predicate<Object>()
                    {
                        @Override
                        public boolean apply(@Nullable Object input)
                        {
                            if (input.toString().startsWith("_"))
                                return false;
                            return true;
                        }
                    });
                }
                catch (UniformInterfaceException e)
                {
                    throw convertToServiceException(e);
                }
            }
        });
    }


    protected <E> Promise<? extends Iterable<E>> getFutureGenericCollection(final WebResource resource, final Class<? extends Iterable> collectionType, final Class<E> contentType)
    {
        return executor.submit(new Callable<Iterable<E>>()
        {
            @Override
            public Iterable<E> call() throws Exception
            {
                return getGenericIterable(resource, collectionType, contentType);
            }
        });
    }

    protected <E> Promise<E> postFuture(final WebResource resource, final Class<? extends E> entityClass, final Object postEntity)
    {
        return postFuture(resource, entityClass, postEntity, MediaType.APPLICATION_JSON_TYPE);
    }

    protected <E> Promise<E> postFuture(final WebResource resource, final Class<? extends E> entityClass, final Object postEntity, final MediaType mediaType)
    {
        return executor.submit(new Callable<E>()
        {
            @Override
            public E call() throws Exception
            {
                return post(resource, entityClass, postEntity, mediaType);
            }
        });
    }

    protected <E> Promise<E> postFuture(final WebResource resource, final Class<E> collectionClass, final Class contentType, final Object postEntity)
    {
        return executor.submit(new Callable<E>()
        {
            @Override
            public E call() throws Exception
            {
                return post(resource, collectionClass, contentType, postEntity);
            }
        });
    }

	protected <T> Promise<PageResponse<T>> postFutureToPageResponse(final WebResource resource, final Class<T> entityClass, final Object postEntity, final MediaType mediaType)
    {
        return executor.submit(new Callable<PageResponse<T>>()
        {
            @Override
            public PageResponse<T> call() throws Exception
            {
                return postToPageResponse(resource, entityClass, postEntity, mediaType);
            }
        });
    }

    protected <T> Promise<T> putFuture(final WebResource resource, final Class<? extends T> entityClass, final Object putEntity)
    {
        return executor.submit(new Callable<T>()
        {
            @Override
            public T call() throws Exception
            {
                return put(resource, entityClass, putEntity);
            }
        });
    }

    protected Promise<Void> deleteFuture(final WebResource resource)
    {
        return deleteFuture(resource, Void.class);
    }

    protected <T> Promise<T> deleteFuture(final WebResource resource, final Class<? extends T> responseClass)
    {
        return executor.submit(new Callable<T>()
        {
            @Override
            public T call() throws Exception
            {
                return delete(resource, responseClass);
            }
        });
    }

    /**
     * add the expansions to the WebResource as a query param, this specifies which properties
     * on the results to expand
     * @param resource - the WebResource to add the query param to, a copy of this resource will be returned
     * @param expansions - the expansions to add
     * @return a new WebResource with the expansions added to it
     */
    protected WebResource addExpansions(WebResource resource, Expansion[] expansions)
    {
        if (expansions.length == 0)
            return resource;

        return resource.queryParam("expand", ExpansionsParser.asString(expansions));
    }

    /**
     * Adds start and limit query params to honour the pageRequest values
     * @param resource - the resource to add the query params to
     * @param request - the pageRequest supplying the values for the query params, may be null
     */
    protected WebResource addPageRequestParams(WebResource resource, @Nullable PageRequest request)
    {
        if (request == null)
            return resource;

        return resource
                .queryParam("start", Integer.toString(request.getStart()))
                .queryParam("limit", Integer.toString(request.getLimit()));
    }

    private <E> Option<E> getOption(WebResource resource, Class<? extends E> clazz) throws ServiceException
    {
        try
        {
            E entity = resource.get(clazz);
            return Option.some(entity);
        }
        catch (UniformInterfaceException e)
        {
            if (e.getResponse().getClientResponseStatus() == ClientResponse.Status.NOT_FOUND)
            {
                log.debug("404 being converted to Option.none : {}", e.getMessage());
                return Option.none();
            }
            else
                throw convertToServiceException(e);
        }
    }

    private <E> PageResponse<E> getPartialList(WebResource resource, final Class<E> contentClass) throws ServiceException
    {
        return (PageResponse<E>)getGenericIterable(resource, RestList.class, contentClass);
    }

    private <G> GenericType<G> getGenericType(final Class<G> collectionType, final Type... typeParameters)
    {
        return new GenericType<G>(getParameterizedType(collectionType, typeParameters));
    }

    private static ParameterizedType getParameterizedType(final Class type, final Type... typeParameters)
    {
        return new ParameterizedType()
        {

            @Override
            public Type[] getActualTypeArguments()
            {
                return typeParameters;
            }

            @Override
            public Type getRawType()
            {
                return type;
            }

            @Override
            public Type getOwnerType()
            {
                return null;
            }
        };
    }

    @SuppressWarnings("unchecked")
    private <T> Iterable<T> getGenericIterable(WebResource resource, final Class<? extends Iterable> collectionType, final Class<T> contentClass) throws ServiceException
    {
        try
        {
            return resource.get(getGenericType(collectionType, contentClass));
        }
        catch (UniformInterfaceException e)
        {
            throw convertToServiceException(e);
        }
    }

	private <T> PageResponse<T> postToPageResponse(WebResource resource, final Class<T> contentClass, Object postEntity, MediaType mediaType) throws ServiceException
    {
        try
        {
            ParameterizedType restListType = new ParameterizedType()
            {

                @Override
                public Type[] getActualTypeArguments()
                {
                    return new Type[]{contentClass};
                }

                @Override
                public Type getRawType()
                {
                    return RestList.class;
                }

                @Override
                public Type getOwnerType()
                {
                    return null;
                }
            };

            WebResource.Builder builder = resource.type(mediaType);
            setAtlassianTokenIfNeeded(builder, mediaType);

            return builder.post(new GenericType<RestList<T>>(restListType), postEntity);
        }
        catch (UniformInterfaceException e)
        {
            throw convertToServiceException(e);
        }
    }

    private <T> T post(WebResource resource, Class<? extends T> clazz, Object postEntity, MediaType mediaType) throws ServiceException
    {
        try
        {
			WebResource.Builder builder = resource.type(mediaType);
            setAtlassianTokenIfNeeded(builder, mediaType);
            return builder.post(clazz, postEntity);
        }
        catch (UniformInterfaceException e)
        {
            throw convertToServiceException(e);
        }
    }

    private <T> T post(WebResource resource, final Class<T> collectionType, final Class contentClass, Object postEntity) throws ServiceException
    {
        try
        {
            GenericType<T> type = getGenericType(collectionType, contentClass);
            return resource.type(MediaType.APPLICATION_JSON_TYPE).post(type, postEntity);
        }
        catch (UniformInterfaceException e)
        {
            throw convertToServiceException(e);
        }

    }

    private void setAtlassianTokenIfNeeded(WebResource.Builder builder, MediaType mediaType)
    {
        if (mediaType != MediaType.APPLICATION_JSON_TYPE)
            builder.header("X-Atlassian-Token", "nocheck");
    }

    private <T> T put(WebResource resource, Class<? extends T> clazz, Object putEntity) throws ServiceException
    {
        try
        {
            // Assumed that all PUTs are JSON.
            return resource.type(MediaType.APPLICATION_JSON_TYPE).put(clazz, putEntity);
        }
        catch (UniformInterfaceException e)
        {
            throw convertToServiceException(e);
        }
    }

    private <T> T delete(WebResource resource, Class<? extends T> clazz) throws ServiceException
    {
        try
        {
            // It's assumed that all DELETEs are JSON.
            WebResource.Builder builder = resource.type(MediaType.APPLICATION_JSON_TYPE);

            if (clazz == Void.class)
            {
                builder.delete();
                return null;
            }

            return builder.delete(clazz);
        }
        catch (UniformInterfaceException e)
        {
            throw convertToServiceException(e);
        }
    }

    protected WebResource addPageRequest(WebResource resource, PageRequest pageRequest)
    {
        return addPageRequestParams(resource, pageRequest);
    }
}
