package com.atlassian.confluence.rest.client.proxy;

import com.atlassian.confluence.rest.client.AbstractRemoteService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Proxy;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

/**
 * Proxy's a remote service with a local service interface.  It is useful in unit
 * tests but that is about all.  Proxies will completely ignore that they are calling over network.  Each
 * call to the proxied service will wait on the returned future for up to 30 seconds
 */
public class RemoteServiceProxyCreator
{
    private static final Logger log = LoggerFactory.getLogger(RemoteServiceProxyCreator.class);
    private static final int timeoutSecs = 30;
    public static <T> T createProxy(final AbstractRemoteService<T> remoteService)
    {
        ParameterizedType parameterizedType = (ParameterizedType) remoteService.getClass().getGenericSuperclass();

        Class<T> proxyClass = (Class<T>) parameterizedType.getActualTypeArguments()[0];
        if (proxyClass == null)
        {
            throw new IllegalArgumentException("RemoteService to be proxied does not supply generic local service type to AbstractRemoteService.  " +
                    "Ensure the type parameter on AbstractRemoteService is specified correctly.");
        }
        return proxyClass.cast(Proxy.newProxyInstance(proxyClass.getClassLoader(),
                new Class[]{proxyClass}, createInvocationHandler(remoteService)));
    }

    private static <T> InvocationHandler createInvocationHandler(final AbstractRemoteService<T> remoteService)
    {
        return new InvocationHandler()
        {
            @Override
            public Object invoke(Object proxy, Method method, Object[] args) throws Throwable
            {
                Method remoteMethod;
                try
                {
                    remoteMethod = remoteService.getClass().getMethod(method.getName(), method.getParameterTypes());
                }
                catch (NoSuchMethodException nsme)
                {
                    log.error("Proxied remote service: {} is missing a method: {}", remoteService.getClass().getSimpleName(), method.getName());
                    throw nsme;
                }
                if (Future.class.isAssignableFrom(remoteMethod.getReturnType()))
                {
                    try
                    {
                        Future future = (Future) remoteMethod.invoke(remoteService, args);
                        return future.get(timeoutSecs, TimeUnit.SECONDS);
                    }
                    catch (ExecutionException ex)
                    {
                        // unwrap the exception, only want the proxy seen if it is getting in the way
                        throw ex.getCause();
                    }
                }
                else if (remoteMethod.getReturnType().equals(method.getReturnType()))
                {
                    return remoteMethod.invoke(remoteService, args);
                }
                else if(returnTypeIsInnerInterface(method) && returnTypeIsInnerInterface(remoteMethod))
                {
                    // if both the the localService and remoteService return inner interfaces, attempt to proxy the result as a remoteService
                    Object potentialRemoteService = remoteMethod.invoke(remoteService, args);
                    if (potentialRemoteService instanceof AbstractRemoteService)
                        return createProxy((AbstractRemoteService)potentialRemoteService);

                    throw new NoSuchMethodException("Remote service "+remoteService.getClass().getSimpleName()+" has a matching method ("+method.getName()+") but the result does not extend AbstractRemoteService: "+potentialRemoteService);

                }
                else
                {
                    throw new NoSuchMethodException(String.format("Remote method : %s has incompatible return type %s with proxy service method return type %s", remoteMethod.getName(),
                            remoteMethod.getReturnType(), method.getReturnType()));
                }
            }
        };
    }

    private static boolean returnTypeIsInnerInterface(Method method)
    {
        Class methodReturnType = method.getReturnType();
        if ( methodReturnType.isInterface())
        {
            Class interfaceDeclarer = methodReturnType.getEnclosingClass();
            // if the return type is not declared in an interface then it is not an inner interface of the interfaces of the method
            if (interfaceDeclarer == null)
                return false;

            Class declaringClass = method.getDeclaringClass();
            do
            {
                if (interfaceDeclarer.isAssignableFrom(declaringClass))
                    return true;
                declaringClass = declaringClass.getEnclosingClass();

            } while (declaringClass != null);

        }
        return false;
    }
}
