package com.atlassian.confluence.rest.client;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.confluence.api.model.Depth;
import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.content.Content;
import com.atlassian.confluence.api.model.content.ContentType;
import com.atlassian.confluence.api.model.content.id.ContentId;
import com.atlassian.confluence.api.model.pagination.PageRequest;
import com.atlassian.confluence.api.model.pagination.PageResponse;
import com.atlassian.confluence.api.service.content.ChildContentService;
import com.atlassian.util.concurrent.Promise;

import java.util.Map;

/**
 * {@link ChildContentService} implementation that communicates with Confluence remotely using the Confluence REST API.
 *
 * Provides future returning equivalents for the methods in ChildContentService.
 */
@ExperimentalApi
public interface RemoteChildContentService
{
    /**
     * Create a finder to find content in the given parent.  Content will be returned
     * upon calling one of the fetch methods on the returned finder
     *
     * @param parentId - the id of the parent to fetch content for
     * @param expansion - the expansions to apply to the content
     * @return a new child content finder with the parent and expansions set
     */
    RemoteChildContentFinder findContent(ContentId parentId, Expansion... expansion);

    interface RemoteChildContentFinder
    {
        /**
         * Retrieve the content contained in this parent.
         * <p>
         * The pageRequest limit is restricted to a maximum page size.
         *
         * @return a map of the content in this parent by contentType
         */
        Promise<Map<ContentType, PageResponse<Content>>> fetchMappedByType(PageRequest request);

        /**
         * Retrieve a paged list of content of the given type contained in this parent
         *
         * @param type - the type of content to retrieve
         * @param request - a page request indicating the start and limit of items to return
         * @return a paged list of content of the given type contained in this parent
         */
        Promise<PageResponse<Content>> fetchMany(ContentType type, PageRequest request);

        /**
         * Set the depth to retrieve content on the ChildContentFinder
         *
         * @param depth - the depth in the page hierarchy of Content descendants to retrieve
         *
         * @return a child content finder with the depth parameter set
         */
        RemoteChildContentFinder withDepth(Depth depth);
    }
}
