package com.atlassian.confluence.rest.client.authentication;

import com.atlassian.annotations.ExperimentalApi;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.filter.HTTPBasicAuthFilter;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;
import java.net.URI;

/**
 * Provides authenticated {@link WebResource}s for Remote API clients.
 *
 * Clients should call {@link #setAuthContext(String, char[])} to set the active user before making remote requests,
 * and {@link #clearAuthContext()} when the user authentication is no longer needed.
 */
@ExperimentalApi
public class AuthenticatedWebResourceProvider
{
    private final String restApiBaseUrl;
    private final Client client;

    private String username;
    private char[] password;

    public AuthenticatedWebResourceProvider(Client client, String baseUrl, String path)
    {
        this.client = client;
        if (baseUrl.endsWith("/"))
            baseUrl = baseUrl.substring(0, baseUrl.length() -1);

        this.restApiBaseUrl = baseUrl + path;
    }

    public WebResource newRestWebResource()
    {
        URI uri = UriBuilder.fromUri(restApiBaseUrl).build();
        WebResource resource = client.resource(uri);
        if (username != null)
        {
            assert password != null;
            resource.addFilter(new HTTPBasicAuthFilter(username, new String(password)));
        }
        return resource;
    }

    /**
     * Returns a new web resource accepting JSON.
     */
    public WebResource.Builder newJsonResource(String path)
    {
        return newRestWebResource().path(path).type(MediaType.APPLICATION_JSON);
    }

    /**
     * Authenticates a user to make remote requests with.
     *
     * @param username the login username of the user
     * @param password the password of the user
     */
    public void setAuthContext(String username, char[] password)
    {
        this.username = username;

        // Alf-hassed security; will need token/cookie after login in production client
        if (this.password != null)
        {
            for (int i = 0; i < this.password.length; i++)
            {
                this.password[i] = (char)0;
            }
        }
        this.password = password;
    }

    /**
     * Clears the current authenticated user, if any.
     */
    public void clearAuthContext()
    {
        setAuthContext(null, null);
    }

    public String getUsername()
    {
        return username;
    }
}
