package com.atlassian.confluence.rest.client;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.confluence.api.model.Depth;
import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.content.Content;
import com.atlassian.confluence.api.model.content.ContentType;
import com.atlassian.confluence.api.model.content.Space;
import com.atlassian.confluence.api.model.longtasks.LongTaskSubmission;
import com.atlassian.confluence.api.model.pagination.PageRequest;
import com.atlassian.confluence.api.model.pagination.PageResponse;
import com.atlassian.confluence.api.service.content.SpaceService;
import com.atlassian.confluence.api.service.exceptions.NotFoundException;
import com.atlassian.confluence.api.service.exceptions.PermissionException;
import com.atlassian.confluence.api.service.exceptions.ServiceException;
import com.atlassian.fugue.Option;
import com.atlassian.util.concurrent.Promise;

import java.util.Map;

import static com.atlassian.confluence.api.service.content.SpaceService.Validator;

/**
 * {@link SpaceService} implementation that communicates with Confluence remotely using the Confluence REST api.
 *
 * Provides future returning equivalents for the methods in SpaceService.
 */
@ExperimentalApi
public interface RemoteSpaceService
{
    /**
     * Create a new space.
     * <p>
     * Minimum properties to be valid is a key and name.
     * </p>
     *
     * @param newSpace the space to create
     * @param isPrivate true if the space should only be visible to its creator
     * @return the space created
     * @throws ServiceException if the space cannot be created
     */
    Promise<Space> create(Space newSpace, boolean isPrivate) throws ServiceException;

    /**
     * Updates a space.
     *
     * <p>
     *     Currently limited to changing the space name, description and homepage.
     * </p>
     *
     * @param space the updated {@link Space}
     * @return the updated space after being persisted
     */
    Promise<Space> update(Space space) throws ServiceException;

    /**
     * Get the validator view of the Space Service.
     */
    Validator validator();

    /**
     * @deprecated since 5.6. Use {@link #find(Expansion...)}
     */
    @Deprecated
    Promise<Option<Space>> getSpace(String spaceKey, Expansion... expansions);

    RemoteSpaceFinder find(Expansion... expansions);

    interface RemoteSpaceFinder extends RemoteSpaceSingleFetcher
    {
        RemoteSpaceFinder withKeys(String... keys);

        Promise<PageResponse<Space>> fetchMany(PageRequest request);
    }

    interface RemoteSpaceSingleFetcher
    {
        Promise<Option<Space>> fetchOne();

        Promise<Space> fetchOneOrNull();
    }

    /**
     * Create a finder to find content in the given space.  Content will be returned
     * upon calling one of the fetch methods on the returned finder
     * @param space - the space to fetch content for
     * @param expansion - the expansions to apply to the content in this space
     * @return a new space content finder with the space and expansions set
     */
    RemoteSpaceContentFinder findContent(Space space, Expansion... expansion);

    interface RemoteSpaceContentFinder
    {
        /**
         * Retrieve the all content contained in this space.
         * <p>
         * The pageRequest limit is restricted to a maximum page size.
         *
         * @return a map of the content in this space by contentType, eith Page or BlogPost
         */
        Promise<Map<ContentType, PageResponse<Content>>> fetchMappedByType(PageRequest request);

        /**
         * Retrieve a paged list of content of the given type contained in this space
         *
         * @param type - the type of content to retrieve, either ContentType.Page or ContentType.BlogPost
         * @param request - a page request indicating the start and limit of items to return
         * @return a paged list of content of the given type contained in this space
         */
        Promise<PageResponse<Content>> fetchMany(ContentType type, PageRequest request);

        /**
         * Set the depth to retrieve content on the SpaceContentFinder
         *
         * @param depth - the depth in the page heirarchy of Content decendents to retrieve, a depth of 0 returns top level content in the space
         *
         * @return a space content finder with the depth parameter set
         */
        RemoteSpaceContentFinder withDepth(Depth depth);
    }

    /**
     * Delete the given space.
     *
     * <p>
     *     The deletion happens asynchronously so a LongTaskSubmission is returned, that can be used to request status.
     * </p>
     *
     * @param spaceToDelete the space to delete
     * @return a LongTaskStatus describing the space-removal operation
     * @throws NotFoundException if the space does not exist or you do not have permission to view it
     * @throws PermissionException if you do not have permission to delete the space
     */
    Promise<LongTaskSubmission> delete(Space spaceToDelete);
}
