package com.atlassian.confluence.rest.client;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.confluence.api.model.content.Content;
import com.atlassian.confluence.api.model.content.ContentType;
import com.atlassian.confluence.api.service.content.ContentTrashService;
import com.atlassian.util.concurrent.Promise;

/**
 * {@link ContentTrashService} implementation that communicates with Confluence remotely using the Confluence REST API.
 *
 * Provides Promise-returning equivalents for the methods in ContentTrashService.
 */
@ExperimentalApi
public interface RemoteContentTrashService
{
    /**
     * Trash a piece of content.
     *
     * <p>
     *     The {@link #purge(Content)} method can be used for content that doesn't support trashing.
     *     Currently, only content with types {@link ContentType#PAGE} and {@link ContentType#BLOG_POST} support trashing.
     * </p>
     *
     * @param content the content to be trashed
     * @return the content after being trashed
     */
    Promise<Void> trash(Content content);

    /**
     * Restore a trashed piece of content.
     *
     * @param content the content to be restored
     * @return the content after being restored
     */
    Promise<Content> restore(Content content);

    /**
     * Permanently delete a piece of content.
     *
     * <p>There are two main cases:</p>
     * <ul>
     *     <li>
     *         Trashable content (e.g. Pages, BlogPosts) must have status "trashed" to be purged.
     *     </li>
     *     <li>
     *         Non-trashable content (e.g. Comments, Attachment) can be purged with status "current".
     *     </li>
     * </ul>
     *
     * @param content the content to be purged
     */
    Promise<Void> purge(Content content);
}
