package com.atlassian.confluence.rest.client;

import com.atlassian.util.concurrent.Promise;
import com.atlassian.util.concurrent.Promises;
import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * An executor service that returns Promises rather than Futures.
 */
public class PromisingExecutorService implements ExecutorService
{
    private final ListeningExecutorService delegate;

    public PromisingExecutorService(ListeningExecutorService service)
    {
        this.delegate = service;
    }
    
    @Override
    public void shutdown()
    {
        delegate.shutdown();
    }

    @Override
    public List<Runnable> shutdownNow()
    {
        return delegate.shutdownNow();
    }

    @Override
    public boolean isShutdown()
    {
        return delegate.isShutdown();
    }

    @Override
    public boolean isTerminated()
    {
        return delegate.isTerminated();
    }

    @Override
    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException
    {
        return delegate.awaitTermination(timeout, unit);
    }

    @Override
    public <T> Promise<T> submit(Callable<T> task)
    {
        return Promises.forListenableFuture(delegate.submit(task));
    }

    @Override
    public <T> Promise<T> submit(Runnable task, T result)
    {
        return Promises.forListenableFuture(delegate.submit(task, result));
    }

    @Override
    public Promise<?> submit(Runnable task)
    {
        return Promises.forListenableFuture(delegate.submit(task));
    }

    <T> List<Promise<T>> toListOfPromises(List<Future<T>> futures)
    {
        return Lists.transform(futures, new Function<Future<T>, Promise<T>>()
        {
            @Override
            public Promise<T> apply(@Nullable Future<T> input)
            {
                if (input instanceof ListenableFuture)
                    return Promises.forListenableFuture((ListenableFuture)input);
                else
                    return Promises.forFuture(input);
            }
        });
    }

    @Override
    public <T> List invokeAll(Collection<? extends Callable<T>> tasks) throws InterruptedException
    {
        return toListOfPromises(delegate.invokeAll(tasks));
    }

    @Override
    public <T> List invokeAll(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit) throws InterruptedException
    {
        return toListOfPromises(delegate.invokeAll(tasks, timeout, unit));
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks) throws InterruptedException, ExecutionException
    {
        return delegate.invokeAny(tasks);
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException
    {
        return delegate.invokeAny(tasks, timeout, unit);
    }

    @Override
    public void execute(Runnable command)
    {
        delegate.execute(command);
    }
}
