package com.atlassian.confluence.rest.client;

import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.content.Content;
import com.atlassian.confluence.api.model.content.ContentStatus;
import com.atlassian.confluence.api.model.content.ContentType;
import com.atlassian.confluence.api.model.content.Space;
import com.atlassian.confluence.api.model.content.id.ContentId;
import com.atlassian.confluence.api.model.locator.ContentLocator;
import com.atlassian.confluence.api.model.pagination.PageRequest;
import com.atlassian.confluence.api.model.pagination.PageResponse;
import com.atlassian.confluence.api.model.pagination.SimplePageRequest;
import com.atlassian.confluence.api.service.content.ContentService;
import com.atlassian.confluence.api.service.exceptions.BadRequestException;
import com.atlassian.confluence.api.service.exceptions.unchecked.NotImplementedServiceException;
import com.atlassian.confluence.rest.client.authentication.AuthenticatedWebResourceProvider;
import com.atlassian.fugue.Option;
import com.atlassian.util.concurrent.Promise;
import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.core.util.MultivaluedMapImpl;
import org.joda.time.LocalDate;

import javax.annotation.Nullable;
import javax.ws.rs.core.MultivaluedMap;
import java.util.List;
import java.util.Map;

import static com.atlassian.confluence.api.model.content.ContentStatus.CURRENT;
import static com.atlassian.confluence.api.model.content.ContentStatus.TRASHED;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;

/**
 * ContentService implementation that communicates with Confluence remotely.
 */
public class RemoteContentServiceImpl extends AbstractRemoteService<ContentService> implements RemoteContentService
{
    public RemoteContentServiceImpl(AuthenticatedWebResourceProvider provider, ListeningExecutorService executor)
    {
        super(provider, executor);
    }

    @Override
    public RemoteContentFinder find(Expansion... expansions)
    {
        return new RemoteContentFinderImpl(this, expansions);
    }

    public class RemoteContentFinderImpl extends AbstractRemoteService<ContentService.ContentFinder> implements RemoteContentFinder
    {
        private final Expansion[] expansions;

        private ContentId contentId;
        private String spaceKey;
        private List<ContentType> contentTypes = ImmutableList.of(ContentType.PAGE);
        private LocalDate createdDate;
        private String title;
        private ImmutableList<ContentStatus> statuses = ImmutableList.of(ContentStatus.CURRENT);

        RemoteContentFinderImpl(RemoteContentServiceImpl otherService, Expansion... expansions)
        {
            super(otherService);
            this.expansions = expansions;
        }

        @Override
        public RemoteSingleContentFetcher withId(ContentId contentId)
        {
            this.contentId = contentId;
            return this;
        }

        @Override
        public RemoteSingleContentFetcher withLocator(ContentLocator locator)
        {
            spaceKey = locator.getSpaceKey();
            createdDate = locator.getPostingDay();
            contentTypes = newArrayList(locator.getContentTypes());
            title = locator.getTitle();

            return this;
        }

        @Override
        public RemoteParameterContentFinder withSpace(Space... spaces)
        {
            if (spaces.length == 1)
                this.spaceKey = spaces[0].getKey();
            else
                spaceKey = null;

            return this;
        }

        @Override
        public RemoteParameterContentFinder withType(ContentType... types)
        {
            if (types.length > 0)
            {
                contentTypes = newArrayList(types);
            }
            return this;
        }

        @Override
        public RemoteParameterContentFinder withCreatedDate(LocalDate time)
        {
            createdDate = time;
            return this;
        }

        @Override
        public RemoteParameterContentFinder withTitle(String title)
        {
            this.title = title;
            return this;
        }

        @Override
        public RemoteContentFinder withStatus(ContentStatus... status)
        {
            this.statuses = ImmutableList.copyOf(checkNotNull(status));
            return this;
        }

        @Override
        public RemoteContentFinder withAnyStatus()
        {
            this.statuses = ImmutableList.of();
            return this;
        }

        @Override
        public Promise<PageResponse<Content>> fetchMany(ContentType type, PageRequest request)
        {
            WebResource webResource = newContentWebResource();
            if (title != null)
                webResource = webResource.queryParam("title", title);

            if (spaceKey != null)
                webResource = webResource.queryParam("spaceKey", spaceKey);

            if (statuses != null)
            {
                MultivaluedMap params = new MultivaluedMapImpl();
                for (ContentStatus contentStatus: statuses)
                {
                    params.add("status", contentStatus.toString());
                }
                webResource = webResource.queryParams(params);
            }

            webResource = webResource.queryParam("type", type.getType());

            if (createdDate != null)
                webResource = webResource.queryParam("postingDay", createdDate.toString("yyyy-MM-dd"));

            webResource = addExpansions(webResource, expansions);

            webResource = addPageRequest(webResource, request);

            return getFuturePageResponseList(webResource, Content.class);
        }

        @Override
        public Promise<Map<ContentType, PageResponse<Content>>> fetchMappedByContentType(PageRequest request)
        {
            // we'll do this with a request per contentType
            throw new NotImplementedServiceException("fetchMappedByContentType not yet supported");
        }

        @Override
        public Promise<Option<Content>> fetchOne()
        {
            if (contentId != null)
            {
                WebResource webResource = newContentWebResource().path(contentId.serialise());
                webResource = addExpansions(webResource, expansions);
                webResource = handleStatus(webResource);

                return getFutureOption(webResource , Content.class);
            }
            if (contentTypes.isEmpty())
                throw new BadRequestException("Cannot fetch content without specifying a contentType or a contentId");

            return fetchMany(contentTypes.get(0), SimplePageRequest.ONE).map(new Function<PageResponse<Content>, Option<Content>>()
            {

                @Override
                public Option<Content> apply(@Nullable PageResponse<Content> input)
                {
                    if (input.size() > 0)
                        return Option.some(input.iterator().next());
                    return Option.none();
                }
            });
        }

        /**
         * Allowing for ContentStatus filtering via REST is a little interesting. By default, REST GETs for content
         * that don't include a 'status' query parameter only return content with status 'current'.
         *
         * Specifying one or more statuses to filter on does what you'd expect (passes them as query params), but to
         * return content with no status filtering at all means indicating to the server that all statuses should be
         * allowed.
         *
         * We achieve this with a query param of "status=any", where "any" is obviously not a valid status, but is a
         * valid filter.
         */
        private WebResource handleStatus(WebResource webResource)
        {
            if (statuses.isEmpty())
            {
                webResource = webResource.queryParam("status", "any");
            }
            else if (!statuses.equals(ImmutableList.of(CURRENT)))
            {
                for (ContentStatus status : statuses)
                {
                    webResource = webResource.queryParam("status", status.serialise());
                }
            }

            return webResource;
        }

        @Override
        public Promise<Content> fetchOneOrNull()
        {
            return fetchOne().map(new Function<Option<Content>, Content>(){

                @Override
                public Content apply(@Nullable Option<Content> input)
                {
                    return input.getOrNull();
                }
            });
        }
    }

    @Override
    public Promise<Content> create(Content newContent)
    {
        WebResource resource = newContentWebResource();
        return postFuture(resource, Content.class, newContent);
    }

    @Override
    public Promise<Content> update(Content content)
    {
        // Edit path is PUT /content/{id}
        WebResource resource = newContentWebResource().path(content.getId().serialise());
        return putFuture(resource, Content.class, content);
    }

    @Override
    public Promise<Content> trash(Content content)
    {
        // Delete path is DELETE /content/{id} but only works on Pages and Blogposts.
        WebResource resource = newExperimentalContentWebResource().path(content.getId().serialise());
        return deleteFuture(resource, Content.class);
    }

    @Override
    public Promise<Content> restore(Content content)
    {
        // Restore path is PUT /content/{id}, request Content status must be "current".
        if (!CURRENT.equals(content.getStatus()))
        {
            content = Content.builder(content).status(CURRENT).build();
        }
        WebResource resource = newExperimentalContentWebResource().path(content.getId().serialise());
        return putFuture(resource, Content.class, content);
    }

    @Override
    public Promise<Void> purge(Content content)
    {
        // Delete path is DELETE /content/{id}?status=trashed and only works on Pages and Blogposts.
        WebResource resource = newExperimentalContentWebResource().path(content.getId().serialise()).queryParam("status", TRASHED.serialise());
        return deleteFuture(resource);
    }

    @Override
    public Promise<Void> delete(Content content)
    {
        // Delete path is DELETE /content/{id}
        WebResource resource = newContentWebResource().path(content.getId().serialise());
        return deleteFuture(resource);
    }

    @Override
    public Promise<PageResponse<Content>> getChildren(Content parent, PageRequest pageRequest, Expansion... expansions)
    {
        WebResource resource = newContentWebResource().path(parent.getId().serialise()).path("children");
        resource = addExpansions(resource, expansions);

        return getFuturePageResponseList(resource, Content.class);
    }

    public WebResource newContentWebResource()
    {
        return newRestWebResource().path("content");
    }

    public WebResource newExperimentalContentWebResource()
    {
        return newRestWebResource().path("experimental").path("content");
    }

}
