package com.atlassian.confluence.rest.client;

import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.content.AttachmentUpload;
import com.atlassian.confluence.api.model.content.Content;
import com.atlassian.confluence.api.model.content.id.ContentId;
import com.atlassian.confluence.api.model.pagination.PageRequest;
import com.atlassian.confluence.api.model.pagination.PageResponse;
import com.atlassian.confluence.api.model.pagination.SimplePageRequest;
import com.atlassian.confluence.api.service.content.AttachmentService;
import com.atlassian.confluence.api.service.exceptions.ServiceException;
import com.atlassian.confluence.rest.api.model.ExpansionsParser;
import com.atlassian.confluence.rest.client.authentication.AuthenticatedWebResourceProvider;
import com.atlassian.fugue.Option;
import com.atlassian.util.concurrent.Promise;
import com.atlassian.util.concurrent.Promises;
import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.multipart.BodyPart;
import com.sun.jersey.multipart.FormDataMultiPart;
import com.sun.jersey.multipart.file.StreamDataBodyPart;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import javax.ws.rs.core.MediaType;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.Collection;
import java.util.concurrent.Future;

import static com.atlassian.confluence.rest.client.RemoteContentService.RemoteSingleContentFetcher;

/**
 * AttachmentService implementation that communicates with Confluence remotely using the Confluence REST api
 */
public class RemoteAttachmentServiceImpl extends AbstractRemoteService<AttachmentService> implements RemoteAttachmentService
{
    private static final Logger log = LoggerFactory.getLogger(RemoteAttachmentServiceImpl.class);

    private final RemoteContentService remoteContentService;


    public RemoteAttachmentServiceImpl(AuthenticatedWebResourceProvider provider, ListeningExecutorService executor)
    {
        super(provider, executor);
        remoteContentService = new RemoteContentServiceImpl(provider, executor);
    }

    @Override
    public Promise<PageResponse<Content>> addAttachments(ContentId contentId, Collection<AttachmentUpload> uploads) throws ServiceException
    {
        WebResource resource = newAttachmentWebResource(contentId);
        FormDataMultiPart multiPart = makeMultiPart(uploads);
        return postFutureToPageResponse(resource, Content.class, multiPart, MediaType.MULTIPART_FORM_DATA_TYPE);
    }

    private WebResource newAttachmentWebResource(ContentId containerId)
    {
        return newRestWebResource().path("content").path(containerId.serialise()).path("child").path("attachment");
    }

    @Override
    public RemoteAttachmentFinder find(Expansion... expansions)
    {
        return new RemoteAttachmentFinderImpl(this, expansions);
    }

    public class RemoteAttachmentFinderImpl  extends AbstractRemoteService<AttachmentService.AttachmentFinder> implements RemoteAttachmentFinder
    {
        private Expansion[] expansions;

        private ContentId attachmentId;
        private ContentId containerId;
        private String filename;
        private String mediaType;

        RemoteAttachmentFinderImpl(RemoteAttachmentServiceImpl otherService, Expansion[] expansions)
        {
            super(otherService);
            this.expansions = expansions;
        }

        @Override
        public RemoteSingleContentFetcher withId(ContentId attachmentId)
        {
            this.attachmentId = attachmentId;
            return this;
        }

        @Override
        public RemoteAttachmentFinder withContainerId(ContentId containerId)
        {
            this.containerId = containerId;
            return this;
        }

        @Override
        public RemoteAttachmentFinder withFilename(String filename)
        {
            this.filename = filename;
            return this;
        }

        @Override
        public RemoteAttachmentFinder withMediaType(String mediaType)
        {
            this.mediaType = mediaType;
            return this;
        }
        @Override
        public Promise<PageResponse<Content>> fetchMany(PageRequest request)
        {
            WebResource webResource = newAttachmentWebResource(containerId);

            if (filename != null)
                webResource = webResource.queryParam("filename", filename);

            if (mediaType != null)
                webResource = webResource.queryParam("mediaType", mediaType);

            webResource = addExpansions(webResource, expansions);

            webResource = addPageRequest(webResource, request);

            return getFuturePageResponseList(webResource, Content.class);
        }

        @Override
        public Promise<Option<Content>> fetchOne()
        {
            if (attachmentId != null)
            {
                return remoteContentService.find(expansions).withId(attachmentId).fetchOne();
            }

            return fetchMany(SimplePageRequest.ONE).map(new Function<PageResponse<Content>, Option<Content>>()
            {
                @Override
                public Option<Content> apply(PageResponse<Content> input)
                {
                    if (input.size() > 0)
                        return Option.some(input.iterator().next());
                    return Option.none();
                }
            });
        }

        @Override
        public Promise<Content> fetchOneOrNull()
        {
            return fetchOne().map(new Function<Option<Content>, Content>(){

                @Override
                public Content apply(@Nullable Option<Content> input)
                {
                    return input.getOrNull();
                }
            });
        }
    }

    @Override
    public Promise<Content> update(Content attachment) throws ServiceException
    {
        ContentId containerId = ((Content) attachment.getContainer()).getId();
        WebResource resource =  newAttachmentWebResource(containerId).path(attachment.getId().serialise());
        return putFuture(resource, Content.class, attachment);
    }

    @Override
    public Promise<Content> updateData(final ContentId attachmentId, final AttachmentUpload upload) throws ServiceException
    {
        // We need the contentId of the attachment container for the REST resource Path, so we'll need
        // to retrieve the Attachment and its container from the server first. Use a Promise flatMap to
        // wrap the two async calls.
        Promise<Option<Content>> byId = remoteContentService.find(ExpansionsParser.parseSingle("container")).withId(attachmentId).fetchOne();
        return byId.flatMap(
                new Function<Option<Content>, Promise<Content>>()
                {
                    @Override
                    public Promise<Content> apply(Option<Content> oldAttachmentOption)
                    {
                        Content container = (Content) oldAttachmentOption.get().getContainer();

                        WebResource resource = newAttachmentWebResource(container.getId())
                                .path(attachmentId.serialise())
                                .path("data");
                        FormDataMultiPart multiPart = makeMultiPart(Lists.newArrayList(upload));
                        Future<Content> postFuture = postFuture(resource, Content.class, multiPart, MediaType.MULTIPART_FORM_DATA_TYPE);

                        return Promises.forFuture(postFuture);
                    }
                });

    }

    @Override
    public Promise<Void> delete(Content attachmentContent) throws ServiceException
    {
        // A Delete override is not offered on the AttachmentResource - it's no different to deleting Content.
        return remoteContentService.delete(attachmentContent);
    }

    private FormDataMultiPart makeMultiPart(Collection<AttachmentUpload> uploads)
    {
        FormDataMultiPart multiPart = new FormDataMultiPart();

        for (AttachmentUpload upload : uploads)
        {
            multiPart.bodyPart(makeBodyPart(upload));
            String comment = upload.getComment();
            if (comment == null)
            {
                // Comment field array in the MultiPart needs to align with the File array, and Jersey chokes on
                // null FormDataBodyPart entities.
                comment = "";
            }
            multiPart.field("comment", comment);
            multiPart.field("minorEdit", Boolean.toString(upload.isMinorEdit()));
        }

        return multiPart;
    }

    private BodyPart makeBodyPart(AttachmentUpload upload)
    {
        // The MIME Content-Disposition header has quoted-string values that require the filename to be escaped, but the
        // FileDataBodyPart class makes this impractical. Best to fail fast here, than in a cryptic REST 400 response.
        if (upload.getName().contains("\""))
            throw new IllegalArgumentException("Double-quotes in filenames are not supported in the REST client");

        MediaType mediaType;
        String mediaTypeStr = upload.getMediaType();
        try
        {
            mediaType = MediaType.valueOf(mediaTypeStr);
        }
        catch (IllegalArgumentException e)
        {
            // some test author probably thinks that "text" is a valid Media Type. Sad.
            log.warn("Using 'text/plain' because type supplied is not a known media-type: " + mediaTypeStr);
            mediaType = MediaType.TEXT_PLAIN_TYPE;
        }
        try
        {
            return new StreamDataBodyPart("file", new FileInputStream(upload.getFile()), upload.getName(), mediaType);
        }
        catch (FileNotFoundException e)
        {
            throw new IllegalStateException("File " + upload.getFile() + " does not exist");
        }
    }

    /**
     * Remote validation not supported
     * @throws UnsupportedOperationException
     */
    public AttachmentService.Validator validator()
    {
        throw new UnsupportedOperationException("Remote validation is not supported by the client");
    }
}
