package com.atlassian.confluence.rest.client;

import com.atlassian.confluence.api.model.Depth;
import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.content.Content;
import com.atlassian.confluence.api.model.content.ContentType;
import com.atlassian.confluence.api.model.content.Space;
import com.atlassian.confluence.api.model.longtasks.LongTaskSubmission;
import com.atlassian.confluence.api.model.pagination.PageRequest;
import com.atlassian.confluence.api.model.pagination.PageResponse;
import com.atlassian.confluence.api.model.pagination.SimplePageRequest;
import com.atlassian.confluence.api.service.content.SpaceService;
import com.atlassian.confluence.api.service.exceptions.ServiceException;
import com.atlassian.confluence.api.service.exceptions.unchecked.NotImplementedServiceException;
import com.atlassian.confluence.rest.client.authentication.AuthenticatedWebResourceProvider;
import com.atlassian.fugue.Option;
import com.atlassian.util.concurrent.Promise;
import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.core.util.MultivaluedMapImpl;

import java.util.List;
import java.util.Map;

import static com.atlassian.confluence.api.service.content.SpaceService.SpaceFinder;
import static com.google.common.collect.Lists.newArrayList;

/**
 * SpaceService implementation that communicates with Confluence remotely using the Confluence REST api
 * 
 */
public class RemoteSpaceServiceImpl extends AbstractRemoteService<SpaceService> implements RemoteSpaceService
{
    public static final String SPACE_RESOURCE_PATH = "space";
    public static final String PRIVATE_SPACE_SUBPATH = "_private";

    public RemoteSpaceServiceImpl(AuthenticatedWebResourceProvider provider, ListeningExecutorService executor)
    {
        super(provider, executor);
    }

    @Override
    public Promise<Space> create(Space newSpace, boolean isPrivate) throws ServiceException
    {
        WebResource resource = newSpacesRestResource();
        if (isPrivate)
            resource = resource.path(PRIVATE_SPACE_SUBPATH);

        return postFuture(resource, Space.class, newSpace);
    }

    @Override
    public Promise<Space> update(Space space) throws ServiceException
    {
        WebResource resource = newSpacesRestResource(space);
        return putFuture(resource, Space.class, space);
    }

    @Override
    public SpaceService.Validator validator()
    {
        throw new NotImplementedServiceException("SpaceService.validator() not yet supported");
    }

    @Override
    @Deprecated
    public Promise<Option<Space>> getSpace(String spaceKey, Expansion... expansions)
    {
        throw new UnsupportedOperationException("Use find().withKeys(spaceKey).fetchOne() instead.");
    }

    @Override
    public RemoteSpaceFinder find(Expansion... expansions)
    {
        return new RemoteSpaceFinderImpl(this, expansions);
    }

    public class RemoteSpaceFinderImpl extends AbstractRemoteService<SpaceFinder> implements RemoteSpaceFinder
    {
        private List<String> keys = newArrayList();
        private final Expansion[] expansions;

        protected RemoteSpaceFinderImpl(AbstractRemoteService other, Expansion... expansions)
        {
            super(other);
            this.expansions = expansions;
        }

        @Override
        public RemoteSpaceFinder withKeys(String... keys)
        {
            this.keys = ImmutableList.copyOf(keys);
            return this;
        }

        @Override
        public Promise<PageResponse<Space>> fetchMany(PageRequest request)
        {
            WebResource webResource = addExpansions(newSpacesRestResource(), expansions);
            webResource = addPageRequestParams(webResource, request);

            MultivaluedMapImpl params = new MultivaluedMapImpl();
            params.put("spaceKey", keys);

            webResource = webResource.queryParams(params);
            return getFuturePageResponseList(webResource, Space.class);
        }

        @Override
        public Promise<Option<Space>> fetchOne()
        {
            if (keys.size() == 1)
            {
                WebResource resource = addExpansions(newSpacesRestResource(keys.get(0)), expansions);
                return getFutureOption(resource, Space.class);
            }
            return fetchMany(SimplePageRequest.ONE).map(new Function<PageResponse<Space>, Option<Space>>()
            {
                @Override
                public Option<Space> apply(PageResponse<Space> input)
                {
                    if (input.size() > 0)
                        return Option.some(input.iterator().next());
                    return Option.none();
                }
            });
        }

        @Override
        public Promise<Space> fetchOneOrNull()
        {
            return fetchOne().map(new Function<Option<Space>, Space>(){

                @Override
                public Space apply(Option<Space> input)
                {
                    return input.getOrNull();
                }
            });
        }
    }

    @Override
    public RemoteSpaceContentFinder findContent(Space space, Expansion... expansion)
    {
        return new RemoteSpaceContentFinderImpl(this, space, expansion);
    }

    public class RemoteSpaceContentFinderImpl extends AbstractRemoteService<SpaceService.SpaceContentFinder> implements RemoteSpaceContentFinder
    {
        WebResource resource;
        private Depth depth = Depth.ALL;

        public RemoteSpaceContentFinderImpl(AbstractRemoteService other, Space space, Expansion[] expansion)
        {
            super(other);
            resource = newSpacesRestResource(space).path("content");
            resource = addExpansions(resource, expansion);
        }

        @Override
        public Promise<Map<ContentType, PageResponse<Content>>> fetchMappedByType(PageRequest request)
        {
            resource = addPageRequestParams(resource, request);
            resource = addParams();
            return getFutureMapOfPageResponses(resource, ContentType.class, Content.class);

        }

        private WebResource addParams()
        {
            return resource.queryParam("depth", depth.toString());
        }

        @Override
        public Promise<PageResponse<Content>> fetchMany(ContentType type, PageRequest request)
        {
            resource = addPageRequestParams(resource.path(type.getType().toLowerCase()), request);
            resource = addParams();
            return getFuturePageResponseList(resource, Content.class);
        }

        @Override
        public RemoteSpaceContentFinder withDepth(Depth depth)
        {
            this.depth = depth;
            return this;
        }
    }

    @Override
    public Promise<LongTaskSubmission> delete(Space space)
    {
        WebResource resource = newSpacesRestResource(space);
        return deleteFuture(resource, LongTaskSubmission.class);
    }

    private WebResource newSpacesRestResource()
    {
        return newRestWebResource().path(SPACE_RESOURCE_PATH);
    }

    private WebResource newSpacesRestResource(Space space)
    {
        return newSpacesRestResource(space.getKey());
    }

    private WebResource newSpacesRestResource(String spaceKey)
    {
        return newSpacesRestResource().path(spaceKey);
    }
}
