package com.atlassian.confluence.rest.client;

import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.content.Content;
import com.atlassian.confluence.api.model.content.JsonContentProperty;
import com.atlassian.confluence.api.model.content.id.ContentId;
import com.atlassian.confluence.api.model.content.id.JsonContentPropertyId;
import com.atlassian.confluence.api.model.pagination.PageRequest;
import com.atlassian.confluence.api.model.pagination.PageResponse;
import com.atlassian.confluence.api.service.content.ContentPropertyService;
import com.atlassian.confluence.rest.client.authentication.AuthenticatedWebResourceProvider;
import com.atlassian.fugue.Option;
import com.atlassian.util.concurrent.Promise;
import com.google.common.base.Function;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.sun.jersey.api.client.WebResource;

import static com.google.common.base.Preconditions.checkNotNull;

public class RemoteContentPropertyServiceImpl extends AbstractRemoteService<ContentPropertyService> implements RemoteContentPropertyService
{
    public RemoteContentPropertyServiceImpl(AuthenticatedWebResourceProvider provider, ListeningExecutorService executor)
    {
        super(provider, executor);
    }

    @Override
    public RemoteContentPropertyFinder find(Expansion... expansions)
    {
        return new RemoteContentPropertyFinderImpl(this, expansions);
    }

    @Override
    public Promise<JsonContentProperty> create(JsonContentProperty newProperty)
    {
        WebResource resource = getContentPropertyWebResource(newProperty);
        return postFuture(resource, JsonContentProperty.class, newProperty);
    }

    @Override
    public Promise<JsonContentProperty> update(JsonContentProperty property)
    {
        WebResource resource = getContentPropertyWebResource(property).path(property.getKey());
        return putFuture(resource, JsonContentProperty.class, property);
    }

    @Override
    public Promise<Void> delete(JsonContentProperty property)
    {
        WebResource resource = getContentPropertyWebResource(property).path(property.getKey());
        return deleteFuture(resource);
    }

    @Override
    public RemoteValidator validator()
    {
        throw new UnsupportedOperationException("Not implemented");
    }

    private WebResource getContentPropertyWebResource(JsonContentProperty property)
    {
        return getContentPropertyWebResource(getContentId(property));
    }

    private WebResource getContentPropertyWebResource(ContentId contentId)
    {
        return newRestWebResource().path("content").path(contentId.serialise()).path("property");
    }

    private ContentId getContentId(JsonContentProperty property)
    {
        return Content.getContentId(property.getContentRef());
    }

    public class RemoteContentPropertyFinderImpl extends AbstractRemoteService<ContentPropertyService.ContentPropertyFinder> implements RemoteContentPropertyFinder
    {
        private final Expansion[] expansions;
        private ContentId contentId;
        private JsonContentPropertyId contentPropertyId;
        private String key;

        public RemoteContentPropertyFinderImpl(AbstractRemoteService other, Expansion... expansions)
        {
            super(other);
            this.expansions = expansions;
        }

        @Override
        public RemoteSingleContentPropertyFetcher withId(JsonContentPropertyId contentPropertyId)
        {
            this.contentPropertyId = contentPropertyId;
            return this;
        }

        @Override
        public RemoteParameterContentPropertyFinder withContentId(ContentId contentId)
        {
            this.contentId = contentId;
            return this;
        }

        @Override
        public RemoteParameterContentPropertyFinder withKey(String key)
        {
            this.key = key;
            return this;
        }

        @Override
        public Promise<PageResponse<JsonContentProperty>> fetchMany(PageRequest request)
        {
            WebResource resource = getContentPropertyWebResource(contentId);
            resource = addExpansions(resource, expansions);
            resource = addPageRequest(resource, request);
            return getFuturePageResponseList(resource, JsonContentProperty.class);
        }

        @Override
        public Promise<Option<JsonContentProperty>> fetchOne()
        {
            if (contentPropertyId != null)
                throw new UnsupportedOperationException("remote fetch with contentPropertyId is not supported");

            checkNotNull(contentId, "ContentId is required to locate content properties");

            WebResource resource = getContentPropertyWebResource(contentId);
            if (key != null)
                resource = resource.path(key);

            resource = addExpansions(resource, expansions);
            return getFutureOption(resource, JsonContentProperty.class);
        }

        @Override
        public Promise<JsonContentProperty> fetchOneOrNull()
        {
            Promise<Option<JsonContentProperty>> oneOption = fetchOne();
            return oneOption.map(new Function<Option<JsonContentProperty>, JsonContentProperty>()
            {
                @Override
                public JsonContentProperty apply(Option<JsonContentProperty> input)
                {
                    return input.getOrNull();
                }
            });
        }
    }
}
