package com.atlassian.confluence.rest.client;

import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.content.AttachmentUpload;
import com.atlassian.confluence.api.model.content.Content;
import com.atlassian.confluence.api.model.content.id.ContentId;
import com.atlassian.confluence.api.model.pagination.PageRequest;
import com.atlassian.confluence.api.model.pagination.PageResponse;
import com.atlassian.confluence.api.service.content.AttachmentService;
import com.atlassian.confluence.api.service.exceptions.ServiceException;
import com.atlassian.util.concurrent.Promise;

import java.util.Collection;

import static com.atlassian.confluence.rest.client.RemoteContentService.RemoteSingleContentFetcher;

/**
 * * {@link AttachmentService} implementation that communicates with Confluence remotely using the Confluence REST API.
 *
 * Provides future returning equivalents for the methods in AttachmentService.
 */
public interface RemoteAttachmentService
{
    /**
     * Add one or more attachments to a piece of {@link Content}.
     *
     *
     * @param contentId the id of the content to attach to
     * @param uploads the attachments being uploaded
     *
     * @return the attachments added
     */
    Promise<PageResponse<Content>> addAttachments(ContentId contentId, Collection<AttachmentUpload> uploads) throws ServiceException;

    RemoteAttachmentFinder find(Expansion... expansions);

    public interface RemoteAttachmentFinder extends RemoteSingleContentFetcher
    {
        RemoteSingleContentFetcher withId(ContentId attachmentId);

        RemoteAttachmentFinder withContainerId(ContentId parentId);

        RemoteAttachmentFinder withFilename(String filename);

        RemoteAttachmentFinder withMediaType(String type);

        Promise<PageResponse<Content>> fetchMany(PageRequest request);
    }
    /**
     * Updates the non-data parts of an Attachment {@link Content} instance.
     * <p>
     *     This method can be used to update the container, filename, media-type and comment of an Attachment.
     * </p>
     * @param attachment the attachment to update, must include an id
     * @return the updated attachment as stored in the database
     */
    Promise<Content> update(Content attachment) throws ServiceException;

    /**
     * Updates the data part of an Attachment {@link Content} instance.
     * @param attachmentId the id of the attachment to update
     * @param upload the new content of the Attachment
     * @return the updated attachment as stored in the database
     */
    Promise<Content> updateData(ContentId attachmentId, AttachmentUpload upload) throws ServiceException;

    /**
     * Removes an Attachment from the system.
     *
     * @param attachmentContent the content to remove
     * @throws ServiceException if the content cannot be found, or cannot be deleted
     */
    Promise<Void> delete(Content attachmentContent) throws ServiceException;
}
