package com.atlassian.confluence.rest.client;

import com.atlassian.confluence.api.model.Depth;
import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.content.Content;
import com.atlassian.confluence.api.model.content.ContentType;
import com.atlassian.confluence.api.model.content.id.ContentId;
import com.atlassian.confluence.api.model.pagination.PageRequest;
import com.atlassian.confluence.api.model.pagination.PageResponse;
import com.atlassian.confluence.api.service.content.ChildContentService;
import com.atlassian.confluence.api.service.exceptions.unchecked.NotImplementedServiceException;
import com.atlassian.confluence.rest.client.authentication.AuthenticatedWebResourceProvider;
import com.atlassian.util.concurrent.Promise;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.sun.jersey.api.client.WebResource;

import javax.annotation.Nullable;
import java.util.Map;

import static com.atlassian.confluence.api.service.content.ChildContentService.ChildContentFinder;

/**
 * {@link ChildContentService} implementation that communicates with Confluence remotely.
 */
public class RemoteChildContentServiceImpl extends AbstractRemoteService<ChildContentService> implements RemoteChildContentService
{
    public RemoteChildContentServiceImpl(AuthenticatedWebResourceProvider provider, ListeningExecutorService executor)
    {
        super(provider, executor);
    }

    @Override
    public RemoteChildContentFinder findContent(ContentId parentId, Expansion... expansion)
    {
        return new RemoteChildContentFinderImpl(this, parentId, expansion);
    }

    public static class RemoteChildContentFinderImpl extends AbstractRemoteService<ChildContentFinder> implements RemoteChildContentFinder
    {
        private WebResource resource;
        private Depth depth = Depth.ROOT;

        public RemoteChildContentFinderImpl(AbstractRemoteService other, ContentId parentId, Expansion[] expansion)
        {
            super(other);
            resource = newRestWebResource().path("content").path(parentId.serialise());
            resource = addExpansions(resource, expansion);
        }

        @Override
        public Promise<Map<ContentType, PageResponse<Content>>> fetchMappedByType(@Nullable PageRequest request)
        {
            resource = resource.path(getPathEndForDepth());
            resource = addPageRequestParams(resource, request);
            resource = addParams();
            return getFutureMapOfPageResponses(resource, ContentType.class, Content.class);
        }

        @Override
        public Promise<PageResponse<Content>> fetchMany(ContentType type, @Nullable PageRequest request)
        {
            resource = resource.path(getPathEndForDepth());             // e.g. /child
            resource = resource.path(type.getType().toLowerCase());     // e.g. /comment
            resource = addPageRequestParams(resource, request);
            resource = addParams();
            return getFuturePageResponseList(resource, Content.class);
        }

        @Override
        public RemoteChildContentFinder withDepth(Depth depth)
        {
            this.depth = depth;
            return this;
        }

        private String getPathEndForDepth()
        {
            if (depth == Depth.ROOT)
                return "child";

            if (depth == Depth.ALL)
                return "descendant";

            throw new NotImplementedServiceException("Only ROOT and ALL depth is currently supported");
        }

        private WebResource addParams()
        {
            return resource.queryParam("depth", depth.toString());
        }
    }
}
