/*
 * Copyright 2006-2007 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sf.ezmorph.object;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import net.sf.ezmorph.MorphException;
import net.sf.ezmorph.MorpherRegistry;
import net.sf.ezmorph.ObjectMorpher;

/**
 * An all-purpose Morpher that can morph to several classes.<br>
 * Because this Morpher accepts any class and morphs to Object it should not be
 * added to a MorpherRegistry as it may be too generic for some cases and may
 * result in unwanted transformations.
 *
 * @author Andres Almiray <a href="mailto:aalmiray@users.sourceforge.net">aalmiray@users.sourceforge.net</a>
 */
public class SwitchingMorpher implements ObjectMorpher {
    private Map classMap = new HashMap();
    private MorpherRegistry morpherRegistry;

    public SwitchingMorpher(Map classMap, MorpherRegistry morpherRegistry) {
        this.morpherRegistry = morpherRegistry;
        if (classMap == null || classMap.isEmpty()) {
            throw new MorphException("Must specify at least one mapping");
        }
        this.classMap.putAll(classMap);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }

        if (!(obj instanceof SwitchingMorpher)) {
            return false;
        }

        SwitchingMorpher other = (SwitchingMorpher) obj;
        if (classMap.size() != other.classMap.size()) {
            return false;
        }
        for (Object o : classMap.entrySet()) {
            Map.Entry entry = (Map.Entry) o;
            if (!other.classMap.containsKey(entry.getKey())) {
                return false;
            }
            if (!entry.getValue().equals(other.classMap.get(entry.getKey()))) {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode() {
        int result = 17;
        for (Object o : classMap.entrySet()) {
            Map.Entry entry = (Map.Entry) o;
            result = 31 * result + Objects.hashCode(entry.getKey());
            result = 31 * result + Objects.hashCode(entry.getValue());
        }
        return result;
    }

    @Override
    public Object morph(Object value) {
        if (value == null) {
            return null;
        }

        Class target = (Class) classMap.get(value.getClass());
        return morpherRegistry.morph(target, value);
    }

    @Override
    public Class morphsTo() {
        return Object.class;
    }

    @Override
    public boolean supports(Class clazz) {
        return true;
    }
}
