/*
 * Decompiled with CFR 0.152.
 */
package org.jvnet.hudson.plugins.thinbackup.backup;

import com.google.common.base.Throwables;
import edu.umd.cs.findbugs.annotations.NonNull;
import hudson.PluginWrapper;
import hudson.model.ItemGroup;
import hudson.model.Job;
import hudson.model.Run;
import hudson.model.TopLevelItem;
import hudson.util.RunList;
import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import java.util.stream.Stream;
import jenkins.model.Jenkins;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.DirectoryFileFilter;
import org.apache.commons.io.filefilter.FileFileFilter;
import org.apache.commons.io.filefilter.FileFilterUtils;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.RegexFileFilter;
import org.apache.commons.io.filefilter.TrueFileFilter;
import org.jvnet.hudson.plugins.thinbackup.ThinBackupPeriodicWork;
import org.jvnet.hudson.plugins.thinbackup.ThinBackupPluginImpl;
import org.jvnet.hudson.plugins.thinbackup.backup.BackupSet;
import org.jvnet.hudson.plugins.thinbackup.backup.PluginList;
import org.jvnet.hudson.plugins.thinbackup.utils.ExistsAndReadableFileFilter;
import org.jvnet.hudson.plugins.thinbackup.utils.Utils;

public class HudsonBackup {
    private static final Logger LOGGER = Logger.getLogger("hudson.plugins.thinbackup");
    public static final String BUILDS_DIR_NAME = "builds";
    public static final String CONFIGURATIONS_DIR_NAME = "configurations";
    public static final String PROMOTIONS_DIR_NAME = "promotions";
    public static final String MULTIBRANCH_DIR_NAME = "branches";
    public static final String INDEXING_DIR_NAME = "indexing";
    public static final String JOBS_DIR_NAME = "jobs";
    public static final String USERS_DIR_NAME = "users";
    public static final String ARCHIVE_DIR_NAME = "archive";
    public static final String CONFIG_HISTORY_DIR_NAME = "config-history";
    public static final String USERSCONTENTS_DIR_NAME = "userContent";
    public static final String NEXT_BUILD_NUMBER_FILE_NAME = "nextBuildNumber";
    public static final String PLUGINS_DIR_NAME = "plugins";
    public static final String NODES_DIR_NAME = "nodes";
    public static final String CONFIG_XML = "config.xml";
    public static final String XML_FILE_EXTENSION = ".xml";
    public static final String JPI_FILE_EXTENSION = ".jpi";
    public static final String HPI_FILE_EXTENSION = ".hpi";
    public static final String DISABLED_EXTENSION = ".disabled";
    public static final String ZIP_FILE_EXTENSION = ".zip";
    public static final String INSTALLED_PLUGINS_XML = "installedPlugins.xml";
    public static final String CHANGELOG_HISTORY_PLUGIN_DIR_NAME = "changelog-history";
    public static final String SVN_CREDENTIALS_FILE_NAME = "subversion.credentials";
    public static final String SVN_EXTERNALS_FILE_NAME = "svnexternals.txt";
    public static final String COMPLETED_BACKUP_FILE = "backup-completed.info";
    private final ThinBackupPluginImpl plugin;
    private final File hudsonHome;
    private final File backupRoot;
    private final File backupDirectory;
    private final ThinBackupPeriodicWork.BackupType backupType;
    private final Date latestFullBackupDate;
    private Pattern excludedFilesRegexPattern = null;
    private Pattern backupAdditionalFilesRegexPattern = null;
    private ItemGroup<TopLevelItem> hudson;

    public HudsonBackup(ThinBackupPluginImpl plugin, ThinBackupPeriodicWork.BackupType backupType) {
        this(plugin, backupType, new Date(), (ItemGroup<TopLevelItem>)Jenkins.get());
    }

    public HudsonBackup(ThinBackupPluginImpl plugin, ThinBackupPeriodicWork.BackupType backupType, Date date, ItemGroup<TopLevelItem> hudson) {
        boolean dirCreationResult;
        String backupAdditionalFilesRegex;
        this.hudson = hudson;
        this.plugin = plugin;
        this.hudsonHome = plugin.getJenkinsHome();
        String excludedFilesRegex = plugin.getExcludedFilesRegex();
        if (excludedFilesRegex != null && !excludedFilesRegex.trim().isEmpty()) {
            try {
                this.excludedFilesRegexPattern = Pattern.compile(excludedFilesRegex);
            }
            catch (PatternSyntaxException pse) {
                LOGGER.log(Level.SEVERE, String.format("Regex pattern '%s' for excluding files is invalid, and will be disregarded.", excludedFilesRegex), pse);
                this.excludedFilesRegexPattern = null;
            }
        }
        if ((backupAdditionalFilesRegex = plugin.getBackupAdditionalFilesRegex()) != null && !backupAdditionalFilesRegex.trim().isEmpty()) {
            try {
                this.backupAdditionalFilesRegexPattern = Pattern.compile(backupAdditionalFilesRegex);
            }
            catch (PatternSyntaxException pse) {
                LOGGER.log(Level.SEVERE, String.format("Regex pattern '%s' for including additional files to back up, is invalid, and will be disregarded.", backupAdditionalFilesRegex), pse);
                this.backupAdditionalFilesRegexPattern = null;
            }
        }
        this.backupRoot = new File(plugin.getExpandedBackupPath());
        if (!this.backupRoot.exists() && !(dirCreationResult = this.backupRoot.mkdirs())) {
            LOGGER.log(Level.WARNING, "Unable to create following directory: " + this.backupRoot.getAbsolutePath());
        }
        this.latestFullBackupDate = this.getLatestFullBackupDate();
        if (this.latestFullBackupDate == null) {
            LOGGER.info("No previous full backup found, thus creating one.");
            this.backupType = ThinBackupPeriodicWork.BackupType.FULL;
        } else {
            this.backupType = backupType;
        }
        this.backupDirectory = Utils.getFormattedDirectory(this.backupRoot, this.backupType, date);
    }

    public void backup() throws IOException {
        boolean res;
        if (this.backupType == ThinBackupPeriodicWork.BackupType.NONE) {
            String msg = "Backup type must be FULL or DIFF. Backup cannot be performed.";
            LOGGER.severe("Backup type must be FULL or DIFF. Backup cannot be performed.");
            throw new IllegalStateException("Backup type must be FULL or DIFF. Backup cannot be performed.");
        }
        LOGGER.fine(MessageFormat.format("Performing {0} backup.", new Object[]{this.backupType}));
        if (!this.hudsonHome.exists() || !this.hudsonHome.isDirectory()) {
            String msg = "No Hudson directory found. Backup cannot be performed.";
            LOGGER.severe("No Hudson directory found. Backup cannot be performed.");
            throw new FileNotFoundException("No Hudson directory found. Backup cannot be performed.");
        }
        if (!(this.backupDirectory.exists() && this.backupDirectory.isDirectory() || (res = this.backupDirectory.mkdirs()))) {
            String msg = "Could not create backup directory. Backup cannot be performed.";
            LOGGER.severe("Could not create backup directory. Backup cannot be performed.");
            throw new IOException("Could not create backup directory. Backup cannot be performed.");
        }
        this.backupGlobalXmls();
        this.backupJobs();
        this.backupRootFolder(USERS_DIR_NAME);
        this.backupNodes();
        if (this.plugin.isBackupUserContents()) {
            this.backupRootFolder(USERSCONTENTS_DIR_NAME);
        }
        if (this.plugin.isBackupConfigHistory()) {
            this.backupRootFolder(CONFIG_HISTORY_DIR_NAME);
        }
        if (this.plugin.isBackupPluginArchives()) {
            this.backupPluginArchives();
        }
        try {
            this.storePluginListIfChanged();
        }
        catch (IOException e) {
            if (this.plugin.isFailFast()) {
                throw e;
            }
            LOGGER.warning("Failed to store plugin list changes: " + e.getLocalizedMessage());
            LOGGER.warning(Throwables.getStackTraceAsString((Throwable)e));
        }
        if (this.plugin.isBackupAdditionalFiles()) {
            this.backupAdditionalFiles();
        }
        this.removeEmptyDirs(this.backupDirectory);
        if (this.backupType == ThinBackupPeriodicWork.BackupType.FULL) {
            this.cleanupDiffs();
            this.moveOldBackupsToZipFile(this.backupDirectory);
            this.removeSuperfluousBackupSets();
        }
        this.touchCompleteFile();
    }

    public void touchCompleteFile() throws IOException {
        File backupCompletedFile = new File(this.backupDirectory.getAbsolutePath(), COMPLETED_BACKUP_FILE);
        FileUtils.touch((File)backupCompletedFile);
    }

    public void removeEmptyDirs(File rootDir) throws IOException {
        try (Stream<Path> walk = Files.walk(rootDir.toPath(), new FileVisitOption[0]);){
            walk.sorted(Comparator.reverseOrder()).map(Path::toFile).filter(File::isDirectory).filter(file -> Objects.requireNonNull(file.list()).length == 0).forEach(file1 -> {
                try {
                    Files.delete(file1.toPath());
                }
                catch (IOException e) {
                    LOGGER.log(Level.WARNING, String.format("Cannot delete Backup directory: %s.", file1.getName()), e);
                }
            });
        }
    }

    private void backupGlobalXmls() throws IOException {
        LOGGER.fine("Backing up global configuration files...");
        IOFileFilter suffixFileFilter = FileFilterUtils.and((IOFileFilter[])new IOFileFilter[]{FileFileFilter.INSTANCE, FileFilterUtils.suffixFileFilter((String)XML_FILE_EXTENSION), this.getFileAgeDiffFilter(), this.getExcludedFilesFilter()});
        try {
            FileUtils.copyDirectory((File)this.hudsonHome, (File)this.backupDirectory, (FileFilter)ExistsAndReadableFileFilter.wrapperFilter(suffixFileFilter));
        }
        catch (IOException e) {
            if (this.plugin.isFailFast()) {
                throw e;
            }
            LOGGER.warning("Failed to copy directory: " + e.getLocalizedMessage());
            LOGGER.warning(Throwables.getStackTraceAsString((Throwable)e));
        }
        LOGGER.fine("DONE backing up global configuration files.");
    }

    private void backupJobs() throws IOException {
        LOGGER.fine("Backing up job specific configuration files...");
        File jobsDirectory = new File(this.hudsonHome.getAbsolutePath(), JOBS_DIR_NAME);
        File jobsBackupDirectory = new File(this.backupDirectory.getAbsolutePath(), JOBS_DIR_NAME);
        this.backupJobsDirectory(jobsDirectory, jobsBackupDirectory);
        LOGGER.fine("DONE backing up job specific configuration files.");
    }

    private void backupJobsDirectory(@NonNull File jobsDirectory, File jobsBackupDirectory) throws IOException {
        String[] list = jobsDirectory.list();
        List<String> jobNames = Arrays.asList(list != null ? list : new String[]{});
        LOGGER.log(Level.INFO, "Found " + jobNames.size() + " jobs in " + jobsDirectory.getPath() + " to back up.");
        LOGGER.log(Level.FINE, "\t{0}", jobNames);
        for (String jobName : jobNames) {
            File jobDirectory = new File(jobsDirectory, jobName);
            if (jobDirectory.exists() && jobDirectory.canRead()) {
                if (jobDirectory.isDirectory()) {
                    File childJobsFolder = new File(jobDirectory, JOBS_DIR_NAME);
                    if (childJobsFolder.exists()) {
                        File expectedConfigXml;
                        File folderBackupDirectory = new File(jobsBackupDirectory, jobName);
                        File folderJobsBackupDirectory = new File(folderBackupDirectory, JOBS_DIR_NAME);
                        boolean dirCreationResult = folderJobsBackupDirectory.mkdirs();
                        if (!dirCreationResult) {
                            LOGGER.log(Level.WARNING, "Unable to create following directory during backup creation: " + folderJobsBackupDirectory.getAbsolutePath());
                        }
                        if ((expectedConfigXml = new File(jobDirectory, CONFIG_XML)).exists() && expectedConfigXml.isFile()) {
                            FileUtils.copyFile((File)expectedConfigXml, (File)new File(folderBackupDirectory, CONFIG_XML));
                        }
                        this.backupJobsDirectory(childJobsFolder, folderJobsBackupDirectory);
                        continue;
                    }
                    try {
                        this.backupJob(jobDirectory, jobsBackupDirectory, jobName);
                    }
                    catch (Exception e) {
                        if (this.plugin.isFailFast()) {
                            throw new IOException("Exception in backing up job in directory: " + String.valueOf(jobDirectory), e);
                        }
                        LOGGER.warning("Failed to backup job " + jobName + " correctly: " + e.getLocalizedMessage());
                        LOGGER.warning(Throwables.getStackTraceAsString((Throwable)e));
                    }
                    continue;
                }
                if (!FileUtils.isSymlink((File)jobDirectory)) continue;
                continue;
            }
            String msg = String.format("Either file does not exist or read access denied on directory '%s', cannot back up the job '%s'.", jobDirectory.getAbsolutePath(), jobName);
            LOGGER.severe(msg);
        }
    }

    private void backupJob(File jobDirectory, File jobsBackupDirectory, String jobName) throws IOException, NoSuchFileException, FileNotFoundException {
        File configurationBackupDirectory;
        List<File> configurations;
        File jobBackupDirectory = new File(jobsBackupDirectory, jobName);
        this.backupJobConfigFor(jobDirectory, jobBackupDirectory);
        this.backupBuildsFor(jobDirectory, jobBackupDirectory);
        if (this.isMatrixJob(jobDirectory)) {
            configurations = this.findAllConfigurations(new File(jobDirectory, CONFIGURATIONS_DIR_NAME));
            for (File configurationDirectory : configurations) {
                configurationBackupDirectory = this.createBackupDirectory(jobBackupDirectory, jobDirectory, configurationDirectory);
                this.backupJobConfigFor(configurationDirectory, configurationBackupDirectory);
                this.backupBuildsFor(configurationDirectory, configurationBackupDirectory);
            }
        }
        if (this.isPromotedJob(jobDirectory)) {
            List<File> promotions = this.findAllConfigurations(new File(jobDirectory, PROMOTIONS_DIR_NAME));
            for (File promotionDirectory : promotions) {
                File promotionBackupDirectory = this.createBackupDirectory(jobBackupDirectory, jobDirectory, promotionDirectory);
                this.backupJobConfigFor(promotionDirectory, promotionBackupDirectory);
                this.backupBuildsFor(promotionDirectory, promotionBackupDirectory);
            }
        }
        if (this.isMultibranchJob(jobDirectory)) {
            FileUtils.copyDirectory((File)new File(jobDirectory, INDEXING_DIR_NAME), (File)new File(jobBackupDirectory, INDEXING_DIR_NAME));
            configurations = this.findAllConfigurations(new File(jobDirectory, MULTIBRANCH_DIR_NAME));
            for (File configurationDirectory : configurations) {
                configurationBackupDirectory = this.createBackupDirectory(jobBackupDirectory, jobDirectory, configurationDirectory);
                this.backupJobConfigFor(configurationDirectory, configurationBackupDirectory);
                this.backupBuildsFor(configurationDirectory, configurationBackupDirectory);
            }
        }
    }

    private void backupPluginArchives() throws IOException {
        LOGGER.fine("Backing up actual plugin archives...");
        IOFileFilter pluginArchivesFilter = FileFilterUtils.or((IOFileFilter[])new IOFileFilter[]{FileFilterUtils.suffixFileFilter((String)JPI_FILE_EXTENSION), FileFilterUtils.suffixFileFilter((String)HPI_FILE_EXTENSION)});
        IOFileFilter disabledPluginMarkersFilter = FileFilterUtils.or((IOFileFilter[])new IOFileFilter[]{FileFilterUtils.suffixFileFilter((String)".jpi.disabled"), FileFilterUtils.suffixFileFilter((String)".hpi.disabled")});
        IOFileFilter filter = FileFilterUtils.and((IOFileFilter[])new IOFileFilter[]{FileFileFilter.INSTANCE, FileFilterUtils.or((IOFileFilter[])new IOFileFilter[]{pluginArchivesFilter, disabledPluginMarkersFilter})});
        this.backupRootFolder(PLUGINS_DIR_NAME, filter);
        LOGGER.fine("DONE backing up actual plugin archives.");
    }

    private void backupAdditionalFiles() throws IOException {
        LOGGER.info("Backing up additional files...");
        if (this.backupAdditionalFilesRegexPattern != null) {
            RegexFileFilter addFilesFilter = new RegexFileFilter(this.backupAdditionalFilesRegexPattern);
            IOFileFilter filter = FileFilterUtils.and((IOFileFilter[])new IOFileFilter[]{addFilesFilter, FileFilterUtils.or((IOFileFilter[])new IOFileFilter[]{DirectoryFileFilter.DIRECTORY, FileFilterUtils.and((IOFileFilter[])new IOFileFilter[]{this.getFileAgeDiffFilter(), this.getExcludedFilesFilter()})})});
            try {
                FileUtils.copyDirectory((File)this.hudsonHome, (File)this.backupDirectory, (FileFilter)ExistsAndReadableFileFilter.wrapperFilter(filter));
            }
            catch (IOException e) {
                if (this.plugin.isFailFast()) {
                    throw e;
                }
                LOGGER.warning("Failed to copy directory: " + e.getLocalizedMessage());
                LOGGER.warning(Throwables.getStackTraceAsString((Throwable)e));
            }
        } else {
            LOGGER.info("No Additional File regex was provided: selecting no Additional Files to back up.");
        }
        LOGGER.info("DONE backing up Additional Files.");
    }

    private void backupNodes() throws IOException {
        LOGGER.fine("Backing up nodes configuration files...");
        IOFileFilter filter = FileFilterUtils.nameFileFilter((String)CONFIG_XML);
        try {
            this.backupRootFolder(NODES_DIR_NAME, filter);
        }
        catch (IOException e) {
            if (this.plugin.isFailFast()) {
                throw e;
            }
            LOGGER.warning("Failed to backup nodes configuration folder nodes: " + e.getLocalizedMessage());
            LOGGER.warning(Throwables.getStackTraceAsString((Throwable)e));
        }
        LOGGER.fine("DONE backing up nodes configuration files.");
    }

    private File createBackupDirectory(File jobBackupdirectory, File jobDirectory, File configurationDirectory) {
        String pathToConfiguration = configurationDirectory.getAbsolutePath();
        String pathToJob = jobDirectory.getAbsolutePath();
        return new File(jobBackupdirectory, pathToConfiguration.substring(pathToJob.length()));
    }

    private List<File> findAllConfigurations(File dir) throws UncheckedIOException {
        Collection listFiles = FileUtils.listFiles((File)dir, (IOFileFilter)FileFilterUtils.nameFileFilter((String)CONFIG_XML), (IOFileFilter)TrueFileFilter.INSTANCE);
        ArrayList<File> confs = new ArrayList<File>();
        for (File file : listFiles) {
            confs.add(file.getParentFile());
        }
        return confs;
    }

    private boolean isMatrixJob(File jobDirectory) {
        return new File(jobDirectory, CONFIGURATIONS_DIR_NAME).isDirectory();
    }

    private boolean isPromotedJob(File jobDirectory) {
        return new File(jobDirectory, PROMOTIONS_DIR_NAME).isDirectory();
    }

    private boolean isMultibranchJob(File jobDirectory) {
        return new File(jobDirectory, MULTIBRANCH_DIR_NAME).isDirectory() && new File(jobDirectory, INDEXING_DIR_NAME).isDirectory();
    }

    private void backupJobConfigFor(File jobDirectory, File jobBackupDirectory) throws IOException {
        IOFileFilter filter = FileFilterUtils.and((IOFileFilter[])new IOFileFilter[]{FileFilterUtils.or((IOFileFilter[])new IOFileFilter[]{FileFilterUtils.suffixFileFilter((String)XML_FILE_EXTENSION), FileFilterUtils.nameFileFilter((String)SVN_CREDENTIALS_FILE_NAME), FileFilterUtils.nameFileFilter((String)SVN_EXTERNALS_FILE_NAME)}), this.getFileAgeDiffFilter(), this.getExcludedFilesFilter()});
        FileUtils.copyDirectory((File)jobDirectory, (File)jobBackupDirectory, (FileFilter)ExistsAndReadableFileFilter.wrapperFilter(filter));
        this.backupNextBuildNumberFile(jobDirectory, jobBackupDirectory);
    }

    private void backupNextBuildNumberFile(File jobDirectory, File jobBackupDirectory) throws IOException {
        File nextBuildNumberFile;
        if (this.plugin.isBackupNextBuildNumber() && (nextBuildNumberFile = new File(jobDirectory, NEXT_BUILD_NUMBER_FILE_NAME)).exists()) {
            FileUtils.copyFileToDirectory((File)nextBuildNumberFile, (File)jobBackupDirectory, (boolean)true);
        }
    }

    private void backupBuildsFor(File jobDirectory, File jobBackupDirectory) throws IOException {
        File buildsDir;
        if (this.plugin.isBackupBuildResults() && (buildsDir = new File(jobDirectory, BUILDS_DIR_NAME)).list() != null && buildsDir.exists() && buildsDir.isDirectory()) {
            String[] builds = buildsDir.list();
            TopLevelItem job = (TopLevelItem)this.hudson.getItem(jobDirectory.getName());
            if (builds != null) {
                for (String build : builds) {
                    File source = new File(buildsDir, build);
                    if (this.plugin.isBackupBuildsToKeepOnly() && !this.isBuildToKeep(job, source)) continue;
                    File destDir = new File(new File(jobBackupDirectory, BUILDS_DIR_NAME), build);
                    if (this.isSymLinkFile(source)) continue;
                    this.backupBuildFiles(source, destDir);
                    this.backupBuildArchive(source, destDir);
                }
            }
        }
    }

    private boolean isBuildToKeep(TopLevelItem item, File buildDir) {
        if (item instanceof Job) {
            Job job = (Job)item;
            RunList builds = job.getBuilds();
            for (Run run : builds) {
                if (!run.getRootDir().equals(buildDir)) continue;
                return run.isKeepLog();
            }
        }
        return true;
    }

    private void backupBuildFiles(File source, File destination) throws IOException {
        if (source.isDirectory()) {
            IOFileFilter changelogFilter = FileFilterUtils.and((IOFileFilter[])new IOFileFilter[]{DirectoryFileFilter.DIRECTORY, FileFilterUtils.nameFileFilter((String)CHANGELOG_HISTORY_PLUGIN_DIR_NAME)});
            IOFileFilter fileFilter = FileFilterUtils.and((IOFileFilter[])new IOFileFilter[]{FileFileFilter.INSTANCE, this.getFileAgeDiffFilter()});
            IOFileFilter filter = FileFilterUtils.and((IOFileFilter[])new IOFileFilter[]{FileFilterUtils.or((IOFileFilter[])new IOFileFilter[]{changelogFilter, fileFilter}), this.getExcludedFilesFilter(), FileFilterUtils.notFileFilter((IOFileFilter)FileFilterUtils.suffixFileFilter((String)ZIP_FILE_EXTENSION))});
            FileUtils.copyDirectory((File)source, (File)destination, (FileFilter)ExistsAndReadableFileFilter.wrapperFilter(filter));
        } else if (!FileUtils.isSymlink((File)source) && source.isFile()) {
            FileUtils.copyFile((File)source, (File)destination);
        }
    }

    private void backupBuildArchive(File buildSrcDir, File buildDestDir) throws IOException {
        File archiveSrcDir;
        if (this.plugin.isBackupBuildArchive() && (archiveSrcDir = new File(buildSrcDir, ARCHIVE_DIR_NAME)).isDirectory()) {
            IOFileFilter filter = FileFilterUtils.or((IOFileFilter[])new IOFileFilter[]{FileFilterUtils.directoryFileFilter(), FileFilterUtils.and((IOFileFilter[])new IOFileFilter[]{FileFileFilter.INSTANCE, this.getFileAgeDiffFilter()})});
            FileUtils.copyDirectory((File)archiveSrcDir, (File)new File(buildDestDir, ARCHIVE_DIR_NAME), (FileFilter)ExistsAndReadableFileFilter.wrapperFilter(filter));
        }
    }

    private void backupRootFolder(String folderName) throws IOException {
        try {
            this.backupRootFolder(folderName, TrueFileFilter.INSTANCE);
        }
        catch (IOException e) {
            if (this.plugin.isFailFast()) {
                throw e;
            }
            LOGGER.warning("Failed to backup root folder " + folderName + ": " + e.getLocalizedMessage());
            LOGGER.warning(Throwables.getStackTraceAsString((Throwable)e));
        }
    }

    private void backupRootFolder(String folderName, IOFileFilter fileFilter) throws IOException {
        File srcDirectory = new File(this.hudsonHome.getAbsolutePath(), folderName);
        if (srcDirectory.exists() && srcDirectory.isDirectory()) {
            LOGGER.log(Level.FINE, "Backing up {0}...", folderName);
            File destDirectory = new File(this.backupDirectory.getAbsolutePath(), folderName);
            IOFileFilter filter = FileFilterUtils.and((IOFileFilter[])new IOFileFilter[]{fileFilter, this.getFileAgeDiffFilter(), this.getExcludedFilesFilter()});
            filter = FileFilterUtils.or((IOFileFilter[])new IOFileFilter[]{filter, DirectoryFileFilter.DIRECTORY});
            FileUtils.copyDirectory((File)srcDirectory, (File)destDirectory, (FileFilter)ExistsAndReadableFileFilter.wrapperFilter(filter));
            LOGGER.log(Level.FINE, "DONE backing up {0}.", folderName);
        }
    }

    private boolean isSymLinkFile(File file) throws IOException {
        String canonicalPath = file.getCanonicalPath();
        String absolutePath = file.getAbsolutePath();
        return !canonicalPath.substring(canonicalPath.lastIndexOf(File.separatorChar)).equals(absolutePath.substring(absolutePath.lastIndexOf(File.separatorChar)));
    }

    private void storePluginListIfChanged() throws IOException {
        PluginList pluginList = this.getInstalledPlugins();
        PluginList latestFullPlugins = null;
        if (this.backupType == ThinBackupPeriodicWork.BackupType.DIFF) {
            latestFullPlugins = this.getPluginListFromLatestFull();
        }
        if (latestFullPlugins == null || pluginList.compareTo(latestFullPlugins) != 0) {
            LOGGER.fine("Storing list of installed plugins...");
            pluginList.save();
        } else {
            LOGGER.fine("No changes in plugin list since last full backup.");
        }
        LOGGER.fine("DONE storing list of installed plugins.");
    }

    private PluginList getInstalledPlugins() {
        File pluginVersionList = new File(this.backupDirectory, INSTALLED_PLUGINS_XML);
        PluginList newPluginList = new PluginList(pluginVersionList);
        Jenkins jenkins = Jenkins.get();
        List installedPlugins = jenkins.getPluginManager().getPlugins();
        for (PluginWrapper pluginWrapper : installedPlugins) {
            newPluginList.add(pluginWrapper.getShortName(), pluginWrapper.getVersion());
        }
        return newPluginList;
    }

    private PluginList getPluginListFromLatestFull() throws IOException {
        File latestFullBackupDir = Utils.getFormattedDirectory(this.backupRoot, ThinBackupPeriodicWork.BackupType.FULL, this.latestFullBackupDate);
        File pluginsOfLatestFull = new File(latestFullBackupDir, INSTALLED_PLUGINS_XML);
        PluginList latestFullPlugins = new PluginList(pluginsOfLatestFull);
        latestFullPlugins.load();
        return latestFullPlugins;
    }

    private void removeSuperfluousBackupSets() throws IOException {
        if (this.plugin.getNrMaxStoredFull() > 0) {
            LOGGER.fine("Removing superfluous backup sets...");
            List<BackupSet> validBackupSets = Utils.getValidBackupSets(new File(this.plugin.getExpandedBackupPath()));
            int nrOfRemovedBackups = 0;
            while (validBackupSets.size() > this.plugin.getNrMaxStoredFull()) {
                BackupSet set = validBackupSets.get(0);
                set.delete();
                validBackupSets.remove(set);
                ++nrOfRemovedBackups;
            }
            LOGGER.log(Level.FINE, "DONE. Removed {0} superfluous backup sets.", nrOfRemovedBackups);
        }
    }

    private void cleanupDiffs() throws IOException {
        if (this.plugin.isCleanupDiff()) {
            LOGGER.fine("Cleaning up diffs...");
            List<File> diffDirs = Utils.getBackupTypeDirectories(this.backupDirectory.getParentFile(), ThinBackupPeriodicWork.BackupType.DIFF);
            for (File diffDirToDelete : diffDirs) {
                FileUtils.deleteDirectory((File)diffDirToDelete);
            }
            LOGGER.log(Level.FINE, "DONE. Removed {0} unnecessary diff directories.", diffDirs.size());
        }
    }

    private void moveOldBackupsToZipFile(File currentBackup) {
        if (this.plugin.isMoveOldBackupsToZipFile()) {
            ZipperThread zipperThread = new ZipperThread(this.backupRoot, currentBackup);
            zipperThread.start();
        }
    }

    private IOFileFilter getFileAgeDiffFilter() {
        IOFileFilter result = FileFilterUtils.trueFileFilter();
        if (this.backupType == ThinBackupPeriodicWork.BackupType.DIFF) {
            result = FileFilterUtils.ageFileFilter((Date)this.latestFullBackupDate, (boolean)false);
        }
        return result;
    }

    private IOFileFilter getExcludedFilesFilter() {
        IOFileFilter result = FileFilterUtils.trueFileFilter();
        if (this.excludedFilesRegexPattern != null) {
            result = FileFilterUtils.notFileFilter((IOFileFilter)new RegexFileFilter(this.excludedFilesRegexPattern));
        }
        return result;
    }

    private Date getLatestFullBackupDate() {
        List<File> fullBackups = Utils.getBackupTypeDirectories(this.backupRoot, ThinBackupPeriodicWork.BackupType.FULL);
        if (fullBackups == null || fullBackups.isEmpty()) {
            return null;
        }
        Date result = new Date(0L);
        for (File fullBackup : fullBackups) {
            Date tmp = Utils.getDateFromBackupDirectory(fullBackup);
            if (tmp != null) {
                if (!tmp.after(result)) continue;
                result = tmp;
                continue;
            }
            LOGGER.log(Level.INFO, "Cannot parse directory name ' {0} ', thus ignoring it when getting latest backup date.", fullBackup.getName());
        }
        return result;
    }

    public static class ZipperThread
    extends Thread {
        private static final Logger LOGGER = Logger.getLogger("hudson.plugins.thinbackup");
        private final File backupRoot;
        private final File currentBackup;

        public ZipperThread(File backupRoot, File currentBackup) {
            this.backupRoot = backupRoot;
            this.currentBackup = currentBackup;
        }

        @Override
        public void run() {
            LOGGER.fine("Starting zipper thread...");
            Utils.moveOldBackupsToZipFile(this.backupRoot, this.currentBackup);
            LOGGER.fine("DONE zipping.");
        }
    }
}

