/*
 * The MIT License
 *
 * Copyright 2015 CloudBees, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package org.jenkinsci.plugins.workflow.multibranch;

import edu.umd.cs.findbugs.annotations.NonNull;
import hudson.Extension;
import jenkins.branch.MultiBranchProjectFactory;
import jenkins.branch.MultiBranchProjectFactoryDescriptor;
import jenkins.scm.api.SCMSource;
import jenkins.scm.api.SCMSourceCriteria;
import org.kohsuke.stapler.DataBoundConstructor;
import org.kohsuke.stapler.DataBoundSetter;
import java.io.IOException;

/**
 * Defines organization folders by {@link WorkflowBranchProjectFactory}.
 */
public class WorkflowMultiBranchProjectFactory extends AbstractWorkflowMultiBranchProjectFactory {
    private String scriptPath = WorkflowBranchProjectFactory.SCRIPT;

    public Object readResolve() {
        if (this.scriptPath == null) {
            this.scriptPath = WorkflowBranchProjectFactory.SCRIPT;
        }
        return this;
    }

    @DataBoundSetter
    public void setScriptPath(String scriptPath) {
        if (scriptPath == null || scriptPath.isEmpty()) {
            this.scriptPath = WorkflowBranchProjectFactory.SCRIPT;
        } else {
            this.scriptPath = scriptPath;
        }
    }

    public String getScriptPath() { return scriptPath; }

    @DataBoundConstructor public WorkflowMultiBranchProjectFactory() { }

    @Override protected SCMSourceCriteria getSCMSourceCriteria(@NonNull SCMSource source) {
        return newProjectFactory().getSCMSourceCriteria(source);
    }

    private AbstractWorkflowBranchProjectFactory newProjectFactory() {
        WorkflowBranchProjectFactory workflowBranchProjectFactory = new WorkflowBranchProjectFactory();
        workflowBranchProjectFactory.setScriptPath(scriptPath);
        return workflowBranchProjectFactory;
    }

    @Extension public static class DescriptorImpl extends MultiBranchProjectFactoryDescriptor {

        @Override public MultiBranchProjectFactory newInstance() {
            return new WorkflowMultiBranchProjectFactory();
        }

        @NonNull
        @Override public String getDisplayName() {
            return "Pipeline " + WorkflowBranchProjectFactory.SCRIPT;
        }

    }

    @Override
    protected void customize(WorkflowMultiBranchProject project) throws IOException, InterruptedException {
        project.setProjectFactory(newProjectFactory());
    }
}
