/*
 * The MIT License
 *
 * Copyright 2022 CloudBees, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package org.jenkinsci.plugins.workflow.flow;

import edu.umd.cs.findbugs.annotations.CheckForNull;
import edu.umd.cs.findbugs.annotations.NonNull;
import hudson.ExtensionPoint;
import hudson.model.AbstractDescribableImpl;
import hudson.model.Descriptor;
import java.io.IOException;
import java.io.Serializable;
import org.jenkinsci.plugins.workflow.actions.ErrorAction;
import org.jenkinsci.plugins.workflow.steps.StepContext;
import org.kohsuke.accmod.Restricted;
import org.kohsuke.accmod.restrictions.Beta;

/**
 * User-configurable predicate for errors that may occur during a build.
 * Implementations could check for agent-related outages, for example.
 * Step callers could use a condition to decide whether to ignore or report an error, retry, etc.
 */
@Restricted(Beta.class)
public abstract class ErrorCondition extends AbstractDescribableImpl<ErrorCondition> implements ExtensionPoint, Serializable {

    /**
     * Checks whether a given error matches this condition.
     * @param error some exception thrown during a build
     * @param context the context in which the error is being <em>handled</em>, if available;
     *                {@link ErrorAction#findOrigin} could be used to determine the part of the build in which the error was <em>thrown</em>
     * @return true if the error is recognized
     * @throws IOException as per {@link StepContext#get}
     * @throws InterruptedException as per {@link StepContext#get}
     */
    public abstract boolean test(@NonNull Throwable error, @CheckForNull StepContext context) throws IOException, InterruptedException;

    @Override public ErrorConditionDescriptor getDescriptor() {
        return (ErrorConditionDescriptor) super.getDescriptor();
    }

    public static abstract class ErrorConditionDescriptor extends Descriptor<ErrorCondition> {}

}
