/*
 * The MIT License
 *
 * Copyright 2018 CloudBees, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package org.jenkinsci.plugins.workflow.log;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import hudson.ExtensionList;
import hudson.ExtensionPoint;
import hudson.console.ConsoleLogFilter;
import hudson.console.LineTransformationOutputStream;
import hudson.model.AbstractBuild;
import hudson.model.BuildListener;
import hudson.model.Run;
import hudson.model.TaskListener;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import edu.umd.cs.findbugs.annotations.CheckForNull;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import hudson.Functions;
import java.nio.charset.StandardCharsets;
import jenkins.util.BuildListenerAdapter;
import jenkins.util.JenkinsJVM;
import org.jenkinsci.plugins.workflow.flow.FlowExecutionOwner;
import org.jenkinsci.plugins.workflow.steps.BodyInvoker;
import org.jenkinsci.plugins.workflow.steps.StepContext;
import org.kohsuke.accmod.Restricted;
import org.kohsuke.accmod.restrictions.Beta;

/**
 * A way of decorating output from a {@link TaskListener}.
 * Similar to {@link ConsoleLogFilter} but better matched to Pipeline logging.
 * <p>May be passed to a {@link BodyInvoker} in lieu of {@link BodyInvoker#mergeConsoleLogFilters},
 * using {@link #merge} to pick up any earlier decorator in {@link StepContext#get}.
 * <p>Expected to be serializable either locally or over Remoting,
 * so an implementation of {@link #decorate} cannot assume that {@link JenkinsJVM#isJenkinsJVM}.
 * Any controller-side configuration should thus be saved into instance fields when the decorator is constructed.
 * @see <a href="https://issues.jenkins-ci.org/browse/JENKINS-45693">JENKINS-45693</a>
 */
public abstract class TaskListenerDecorator implements /* TODO Remotable */ Serializable {

    private static final long serialVersionUID = 1;

    private static final Logger LOGGER = Logger.getLogger(TaskListenerDecorator.class.getName());

    /**
     * Apply modifications to a build log.
     * Typical implementations use {@link LineTransformationOutputStream}.
     * @param logger a base logger
     * @return a possibly patched result
     */
    public abstract @NonNull OutputStream decorate(@NonNull OutputStream logger) throws IOException, InterruptedException;

    /**
     * Merges two decorators.
     * @param original the original decorator, if any
     * @param subsequent an overriding decorator, if any
     * @return null, or {@code original} or {@code subsequent}, or a merged result applying one then the other
     */
    public static @Nullable TaskListenerDecorator merge(@CheckForNull TaskListenerDecorator original, @CheckForNull TaskListenerDecorator subsequent) {
        if (original == null) {
            return subsequent;
        } else {
            if (subsequent == null) {
                return original;
            } else {
                return new MergedTaskListenerDecorator(original, subsequent);
            }
        }
    }

    /**
     * Tries to translate a similar core interface into the new API.
     * <p>The filter may implement either {@link ConsoleLogFilter#decorateLogger(AbstractBuild, OutputStream)} and/or {@link ConsoleLogFilter#decorateLogger(Run, OutputStream)},
     * but only {@link ConsoleLogFilter#decorateLogger(AbstractBuild, OutputStream)} will be called, and with a null {@code build} parameter.
     * <p>The filter must be {@link Serializable}, and furthermore must not assume that {@link JenkinsJVM#isJenkinsJVM}:
     * the same constraints as for {@link TaskListenerDecorator} generally.
     * @param filter a filter, or null
     * @return an adapter, or null if it is null or (after issuing a warning) not {@link Serializable}
     * @see <a href="https://github.com/jenkinsci/jep/blob/master/jep/210/README.adoc#backwards-compatibility">JEP-210: Backwards Compatibility</a>
     */
    public static @CheckForNull TaskListenerDecorator fromConsoleLogFilter(@CheckForNull ConsoleLogFilter filter) {
        if (filter == null) {
            return null;
        } else if (filter instanceof Serializable) {
            return new ConsoleLogFilterAdapter(filter);
        } else {
            LOGGER.log(Level.WARNING, "{0} must implement Serializable to be used with Pipeline", filter.getClass());
            return null;
        }
    }

    /**
     * Allows a decorator to be applied to any build.
     * @see #apply
     */
    public interface Factory extends ExtensionPoint {

        /**
         * Supplies a decorator applicable to one build.
         * @param owner a build
         * @return a decorator, optionally
         */
        @CheckForNull TaskListenerDecorator of(@NonNull FlowExecutionOwner owner);

        /**
         *
         * @return boolean, false means to apply step decorators first, then TaskListenerDecorator, true means otherwise
         * @see #apply(TaskListener, FlowExecutionOwner, TaskListenerDecorator)
         */
        @Restricted(Beta.class)
        default boolean isAppliedBeforeMainDecorator(){
            return false;
        }
    }

    /**
     * Wraps a logger in a supplied decorator as well as any available from {@link Factory}s.
     * <p>Does <em>not</em> apply {@link ConsoleLogFilter#all} even via {@link #fromConsoleLogFilter},
     * since there is no mechanical way to tell if implementations actually satisfy the constraints.
     * Anyway these singletons could not determine which build they are being applied to if remoted.
     * @param listener the main logger
     * @param owner a build
     * @param mainDecorator an additional contextual decorator to apply, if any
     * @return a possibly wrapped {@code listener}
     */
    public static BuildListener apply(@NonNull TaskListener listener, @NonNull FlowExecutionOwner owner, @CheckForNull TaskListenerDecorator mainDecorator) {
        JenkinsJVM.checkJenkinsJVM();
        List<TaskListenerDecorator.Factory> decoratorFactories = ExtensionList.lookup(TaskListenerDecorator.Factory.class);
        List<TaskListenerDecorator> decorators = Stream.concat(
                        decoratorFactories.stream().filter(f -> !f.isAppliedBeforeMainDecorator()).map(f -> f.of(owner)),
                        Stream.concat(Stream.of(mainDecorator),
                                decoratorFactories.stream().filter(f -> f.isAppliedBeforeMainDecorator()).map(f -> f.of(owner)))).
                filter(Objects::nonNull).
                collect(Collectors.toCollection(ArrayList::new));
        if (decorators.isEmpty()) {
            return CloseableTaskListener.of(BuildListenerAdapter.wrap(listener), listener);
        } else {
            Collections.reverse(decorators);
            return CloseableTaskListener.of(new DecoratedTaskListener(listener, decorators), listener);
        }
    }

    private static class MergedTaskListenerDecorator extends TaskListenerDecorator {

        private static final long serialVersionUID = 1;

        private final @NonNull TaskListenerDecorator original;
        private final @NonNull TaskListenerDecorator subsequent;

        MergedTaskListenerDecorator(@NonNull TaskListenerDecorator original, @NonNull TaskListenerDecorator subsequent) {
            this.original = original;
            this.subsequent = subsequent;
        }

        @NonNull
        @Override public OutputStream decorate(@NonNull OutputStream logger) throws IOException, InterruptedException {
            // TODO BodyInvoker.MergedFilter probably has these backwards
            return decorateAll(logger, List.of(subsequent, original));
        }

        @Override public String toString() {
            return "MergedTaskListenerDecorator[" + subsequent + ", " + original + "]";
        }

    }

    private static class ConsoleLogFilterAdapter extends TaskListenerDecorator {

        private static final long serialVersionUID = 1;

        @SuppressFBWarnings(value = "SE_BAD_FIELD", justification = "Explicitly checking for serializability.")
        private final @NonNull ConsoleLogFilter filter;

        ConsoleLogFilterAdapter(@NonNull ConsoleLogFilter filter) {
            assert filter instanceof Serializable;
            this.filter = filter;
        }

        @NonNull
        @SuppressWarnings("deprecation") // the compatibility code in ConsoleLogFilter fails to delegate to the old overload when given a null argument
        @Override public OutputStream decorate(@NonNull OutputStream logger) throws IOException, InterruptedException {
            return filter.decorateLogger((AbstractBuild<?, ?>) null, logger);
        }

        @Override public String toString() {
            return "ConsoleLogFilter[" + filter + "]";
        }

    }

    /**
     * Applies a series of decorators in order to a base stream.
     * Catches and logs any errors thrown from {@link #decorate}, skipping that decorator.
     */
    private static OutputStream decorateAll(OutputStream base, List<TaskListenerDecorator> decorators) {
        for (TaskListenerDecorator decorator : decorators) {
            try {
                base = decorator.decorate(base);
            } catch (Throwable x) {
                LOGGER.log(Level.WARNING, null, x);
                Functions.printStackTrace(x, new PrintStream(base, true, StandardCharsets.UTF_8));
            }
        }
        return base;
    }

    private static final class DecoratedTaskListener extends OutputStreamTaskListener.Default implements BuildListener {

        private static final long serialVersionUID = 1;

        /**
         * The listener we are delegating to, which was expected to be remotable.
         * Note that we ignore all of its methods other than {@link TaskListener#getLogger}.
         */
        private final @NonNull TaskListener delegate;

        /**
         * A (nonempty) list of decorators we delegate to.
         * They are applied in reverse order, so the first one has the final say in what gets printed.
         */
        private final @NonNull List<TaskListenerDecorator> decorators;

        private transient OutputStream out;

        DecoratedTaskListener(@NonNull TaskListener delegate, @NonNull List<TaskListenerDecorator> decorators) {
            this.delegate = delegate;
            assert !decorators.isEmpty();
            assert !decorators.contains(null);
            this.decorators = decorators;
        }

        @NonNull
        @Override public OutputStream getOutputStream() {
            if (out == null) {
                out = decorateAll(OutputStreamTaskListener.getOutputStream(delegate), decorators);
            }
            return out;
        }

        @Override public String toString() {
            return "DecoratedTaskListener[" + delegate + decorators + "]";
        }

    }

    private static final class CloseableTaskListener implements BuildListener, AutoCloseable, OutputStreamTaskListener {

        static BuildListener of(BuildListener mainDelegate, TaskListener closeDelegate) {
            if (closeDelegate instanceof AutoCloseable) {
                return new CloseableTaskListener(mainDelegate, closeDelegate);
            } else {
                return mainDelegate;
            }
        }

        private static final long serialVersionUID = 1;

        private final @NonNull TaskListener mainDelegate;
        private final @NonNull TaskListener closeDelegate;

        private CloseableTaskListener(@NonNull TaskListener mainDelegate, @NonNull TaskListener closeDelegate) {
            this.mainDelegate = mainDelegate;
            this.closeDelegate = closeDelegate;
            assert closeDelegate instanceof AutoCloseable;
        }

        @NonNull
        @Override
        public OutputStream getOutputStream() {
            return OutputStreamTaskListener.getOutputStream(mainDelegate);
        }

        @NonNull
        @Override public PrintStream getLogger() {
            return mainDelegate.getLogger();
        }

        @Override public void close() throws Exception {
            ((AutoCloseable) closeDelegate).close();
        }

        @Override public String toString() {
            return "CloseableTaskListener[" + mainDelegate + " / " + closeDelegate + "]";
        }

    }

}
