/*
 * The MIT License
 *
 * Copyright 2014 Jesse Glick.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package org.jenkinsci.plugins.workflow.actions;

import hudson.FilePath;
import hudson.model.Node;
import hudson.model.labels.LabelAtom;
import java.util.Set;
import edu.umd.cs.findbugs.annotations.CheckForNull;
import edu.umd.cs.findbugs.annotations.NonNull;
import jenkins.model.Jenkins;
import org.jenkinsci.plugins.workflow.FilePathUtils;

/**
 * Represents the fact that a step run on a particular workspace.
 */
public abstract class WorkspaceAction implements PersistentAction {

    /** The {@link Node#getNodeName} of the workspace. */
    public abstract @NonNull String getNode();

    /** The {@link FilePath#getRemote} of the workspace. */
    public abstract @NonNull String getPath();

    /**
     * The {@link Node#getAssignedLabels} of the node owning the workspace.
     * {@link Node#getSelfLabel} should be exempted, so this set may be empty in the typical case.
     * (Could be reconstructed in most cases via {@link Jenkins#getNode} on {@link #getNode},
     * but not for an agent which has since been removed, common with clouds.)
     */
    public abstract @NonNull Set<LabelAtom> getLabels();

    /** Reconstructs the live workspace, if possible. */
    public final @CheckForNull FilePath getWorkspace() {
        return FilePathUtils.find(getNode(), getPath());
    }

}
