/* Copyright 2004, 2005, 2006 Acegi Technology Pty Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jenkinsci.plugins.reverse_proxy_auth.auth;

import java.util.Collection;
import java.util.Collections;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.jenkinsci.plugins.reverse_proxy_auth.model.ReverseProxyUserDetails;
import org.springframework.dao.DataAccessException;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.authentication.dao.AbstractUserDetailsAuthenticationProvider;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

/**
 * @author Wilder Rodrigues (wrodrigues@schubergphilis.com)
 */
public class ReverseProxyAuthenticationProvider extends AbstractUserDetailsAuthenticationProvider {
    private static final Logger logger = Logger.getLogger(ReverseProxyAuthenticationProvider.class.getName());

    private ReverseProxyAuthenticator authenticator;
    private ReverseProxyAuthoritiesPopulator authoritiesPopulator;
    private boolean includeDetailsObject = true;

    /**
     * Create an instance with the supplied authenticator and authorities populator implementations.
     *
     * @param authenticator the authentication strategy (bind, password comparison, etc) to be used by
     *     this provider for authenticating users.
     * @param authoritiesPopulator the strategy for obtaining the authorities for a given user after
     *     they've been authenticated.
     */
    public ReverseProxyAuthenticationProvider(
            ReverseProxyAuthenticator authenticator, ReverseProxyAuthoritiesPopulator authoritiesPopulator) {
        setAuthenticator(authenticator);
        setAuthoritiesPopulator(authoritiesPopulator);
    }

    /**
     * Creates an instance with the supplied authenticator and a null authorities populator. In this
     * case, the authorities must be mapped from the user context.
     *
     * @param authenticator the authenticator strategy.
     */
    public ReverseProxyAuthenticationProvider(ReverseProxyAuthenticator authenticator) {
        setAuthenticator(authenticator);
        setAuthoritiesPopulator(new NullAuthoritiesPopulator());
    }

    private void setAuthenticator(ReverseProxyAuthenticator authenticator) {
        Assert.notNull(authenticator, "An Authenticator must be supplied");
        this.authenticator = authenticator;
    }

    private ReverseProxyAuthenticator getAuthenticator() {
        return authenticator;
    }

    private void setAuthoritiesPopulator(ReverseProxyAuthoritiesPopulator authoritiesPopulator) {
        Assert.notNull(authoritiesPopulator, "An AuthoritiesPopulator must be supplied");
        this.authoritiesPopulator = authoritiesPopulator;
    }

    protected ReverseProxyAuthoritiesPopulator getAuthoritiesPopulator() {
        return authoritiesPopulator;
    }

    @Override
    protected void additionalAuthenticationChecks(
            UserDetails userDetails, UsernamePasswordAuthenticationToken authentication)
            throws AuthenticationException {
        // Do not do anything here!

        // if (!userDetails.getPassword().equals(authentication.getCredentials().toString())) {
        //     throw new BadCredentialsException(messages.getMessage(
        //             "AbstractUserDetailsAuthenticationProvider.badCredentials", "Bad credentials"),
        //             includeDetailsObject ? userDetails : null);
        // }
    }

    /**
     * Creates the final {@code UserDetails} object that will be returned by the provider once the
     * user has been authenticated.
     *
     * <p>The {@code ReverseProxyAuthoritiesPopulator} will be used to create the granted authorites
     * for the user.
     *
     * <p>Can be overridden to customize the creation of the final UserDetails instance. The default
     * will merge any additional authorities retrieved from the populator with the propertis of
     * original {@code ldapUser} object and set the values of the username and password.
     *
     * @param user The intermediate LdapUserDetails instance returned by the authenticator.
     * @param username the username submitted to the provider
     * @param password the password submitted to the provider
     * @return The UserDetails for the successfully authenticated user.
     */
    protected UserDetails createUserDetails(ReverseProxyUserDetails user, String username, String password) {
        user.setUsername(username);

        // Hack for now to pass user's own authorities. Will make the full greanted authorities as a
        // bean in the future.
        Collection<? extends GrantedAuthority> extraAuthorities =
                getAuthoritiesPopulator().getGrantedAuthorities(user);
        user.setAuthorities(extraAuthorities);

        return user;
    }

    @Override
    protected UserDetails retrieveUser(String username, UsernamePasswordAuthenticationToken authentication)
            throws AuthenticationException {
        if (!StringUtils.hasLength(username)) {
            throw new BadCredentialsException("Empty Username");
        }

        if (logger.isLoggable(Level.FINE)) {
            logger.fine("Retrieving user " + username);
        }

        String password = (String) authentication.getCredentials();

        // We do not need this when doing Reverse Proxy Auth
        // Assert.notNull(password, "Null password was supplied in authentication token");

        // if (password.length() == 0) {
        //    logger.debug("Rejecting empty password for user " + username);
        //    throw new
        // BadCredentialsException(messages.getMessage("LdapAuthenticationProvider.emptyPassword",
        //            "Empty Password"));
        // }

        try {
            ReverseProxyUserDetails revProxyU = getAuthenticator().authenticate(username, password);

            return createUserDetails(revProxyU, username, password);

        } catch (DataAccessException ldapAccessFailure) {
            throw new AuthenticationServiceException(ldapAccessFailure.getMessage(), ldapAccessFailure);
        }
    }

    public boolean isIncludeDetailsObject() {
        return includeDetailsObject;
    }

    public void setIncludeDetailsObject(boolean includeDetailsObject) {
        this.includeDetailsObject = includeDetailsObject;
    }

    private static class NullAuthoritiesPopulator implements ReverseProxyAuthoritiesPopulator {
        @Override
        public Collection<? extends GrantedAuthority> getGrantedAuthorities(ReverseProxyUserDetails userDetails) {
            return Collections.emptySet();
        }
    }
}
