/*
 * The MIT License
 *
 * Copyright (c) 2016  Michael Bischoff & GeriMedica - www.gerimedica.nl
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package org.jenkinsci.plugins.oic;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.nimbusds.jwt.JWT;
import com.nimbusds.jwt.JWTParser;
import com.nimbusds.oauth2.sdk.GrantType;
import com.nimbusds.oauth2.sdk.auth.ClientAuthenticationMethod;
import com.nimbusds.oauth2.sdk.token.AccessToken;
import com.nimbusds.oauth2.sdk.token.BearerAccessToken;
import com.nimbusds.oauth2.sdk.token.RefreshToken;
import com.nimbusds.openid.connect.sdk.op.OIDCProviderMetadata;
import edu.umd.cs.findbugs.annotations.CheckForNull;
import hudson.Extension;
import hudson.ExtensionList;
import hudson.Util;
import hudson.model.Descriptor;
import hudson.model.Descriptor.FormException;
import hudson.model.Failure;
import hudson.model.Saveable;
import hudson.model.User;
import hudson.security.ChainedServletFilter2;
import hudson.security.SecurityRealm;
import hudson.tasks.Mailer;
import hudson.util.DescribableList;
import hudson.util.FormValidation;
import hudson.util.Secret;
import io.burt.jmespath.Expression;
import io.burt.jmespath.JmesPath;
import io.burt.jmespath.RuntimeConfiguration;
import io.burt.jmespath.jcf.JcfRuntime;
import jakarta.servlet.Filter;
import jakarta.servlet.FilterChain;
import jakarta.servlet.FilterConfig;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InvalidObjectException;
import java.io.ObjectStreamException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.time.Clock;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Random;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import javax.annotation.PostConstruct;
import jenkins.model.IdStrategy;
import jenkins.model.IdStrategyDescriptor;
import jenkins.model.Jenkins;
import jenkins.security.ApiTokenProperty;
import jenkins.security.FIPS140;
import jenkins.security.SecurityListener;
import jenkins.util.SystemProperties;
import org.apache.commons.lang3.StringUtils;
import org.jenkinsci.plugins.oic.properties.AllowedTokenExpirationClockSkew;
import org.jenkinsci.plugins.oic.properties.DisableNonce;
import org.jenkinsci.plugins.oic.properties.DisableTokenVerification;
import org.jenkinsci.plugins.oic.properties.EscapeHatch;
import org.jenkinsci.plugins.oic.properties.LoginQueryParameters;
import org.jenkinsci.plugins.oic.properties.LogoutQueryParameters;
import org.jenkinsci.plugins.oic.properties.Pkce;
import org.kohsuke.accmod.Restricted;
import org.kohsuke.accmod.restrictions.DoNotUse;
import org.kohsuke.accmod.restrictions.NoExternalUse;
import org.kohsuke.stapler.DataBoundConstructor;
import org.kohsuke.stapler.DataBoundSetter;
import org.kohsuke.stapler.Header;
import org.kohsuke.stapler.QueryParameter;
import org.kohsuke.stapler.Stapler;
import org.kohsuke.stapler.StaplerRequest2;
import org.kohsuke.stapler.StaplerResponse2;
import org.kohsuke.stapler.interceptor.RequirePOST;
import org.pac4j.core.context.CallContext;
import org.pac4j.core.context.FrameworkParameters;
import org.pac4j.core.context.WebContext;
import org.pac4j.core.context.session.SessionStore;
import org.pac4j.core.credentials.Credentials;
import org.pac4j.core.exception.TechnicalException;
import org.pac4j.core.exception.http.HttpAction;
import org.pac4j.core.exception.http.RedirectionAction;
import org.pac4j.core.http.callback.NoParameterCallbackUrlResolver;
import org.pac4j.core.profile.creator.ProfileCreator;
import org.pac4j.jee.context.JEEContextFactory;
import org.pac4j.jee.context.JEEFrameworkParameters;
import org.pac4j.jee.context.session.JEESessionStoreFactory;
import org.pac4j.jee.http.adapter.JEEHttpActionAdapter;
import org.pac4j.oidc.client.OidcClient;
import org.pac4j.oidc.config.OidcConfiguration;
import org.pac4j.oidc.credentials.authenticator.OidcAuthenticator;
import org.pac4j.oidc.profile.OidcProfile;
import org.pac4j.oidc.redirect.OidcRedirectionActionBuilder;
import org.springframework.security.authentication.AnonymousAuthenticationToken;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.crypto.bcrypt.BCrypt;
import org.springframework.util.Assert;
import org.springframework.web.util.UriComponentsBuilder;

/**
 * Login with OpenID Connect / OAuth 2
 *
 * @author Michael Bischoff
 * @author Steve Arch
 */
public class OicSecurityRealm extends SecurityRealm {

    private static final Logger LOGGER = Logger.getLogger(OicSecurityRealm.class.getName());
    private IdStrategy userIdStrategy;
    private IdStrategy groupIdStrategy;

    public static enum TokenAuthMethod {
        client_secret_basic(ClientAuthenticationMethod.CLIENT_SECRET_BASIC),
        client_secret_post(ClientAuthenticationMethod.CLIENT_SECRET_POST);

        private ClientAuthenticationMethod clientAuthMethod;

        TokenAuthMethod(ClientAuthenticationMethod clientAuthMethod) {
            this.clientAuthMethod = clientAuthMethod;
        }

        ClientAuthenticationMethod toClientAuthenticationMethod() {
            return clientAuthMethod;
        }
    }

    private static final String ID_TOKEN_REQUEST_ATTRIBUTE = "oic-id-token";
    private static final String NO_SECRET = "none";
    private static final String SESSION_POST_LOGIN_REDIRECT_URL_KEY = "oic-redirect-on-login-url";

    private final String clientId;
    private final Secret clientSecret;

    /** @deprecated see {@link OicServerWellKnownConfiguration#getWellKnownOpenIDConfigurationUrl()} */
    @Deprecated
    private transient String wellKnownOpenIDConfigurationUrl;

    /** @deprecated see {@link OicServerManualConfiguration#getTokenServerUrl()} */
    @Deprecated
    private transient String tokenServerUrl;

    /** @deprecated see {@link OicServerManualConfiguration#getJwksServerUrl()} */
    @Deprecated
    private transient String jwksServerUrl;

    /** @deprecated see {@link OicServerManualConfiguration#getTokenAuthMethod()} */
    @Deprecated
    private transient TokenAuthMethod tokenAuthMethod;

    /** @deprecated see {@link OicServerManualConfiguration#getAuthorizationServerUrl()} */
    @Deprecated
    private transient String authorizationServerUrl;

    /** @deprecated see {@link OicServerManualConfiguration#getUserInfoServerUrl()} */
    @Deprecated
    private transient String userInfoServerUrl;

    private String userNameField = "sub";
    private transient Expression<Object> userNameFieldExpr = null;
    private String tokenFieldToCheckKey = null;
    private transient Expression<Object> tokenFieldToCheckExpr = null;
    private String tokenFieldToCheckValue = null;
    private String fullNameFieldName = null;
    private transient Expression<Object> fullNameFieldExpr = null;
    private String emailFieldName = null;
    private transient Expression<Object> emailFieldExpr = null;
    private String groupsFieldName = null;
    private transient Expression<Object> groupsFieldExpr = null;
    private transient Expression<Object> avatarFieldExpr = null;
    private transient String simpleGroupsFieldName = null;
    private transient String nestedGroupFieldName = null;

    /** @deprecated see {@link OicServerManualConfiguration#getScopes()} */
    @Deprecated
    private transient String scopes = null;

    private final boolean disableSslVerification;
    private boolean logoutFromOpenidProvider = true;

    /** @deprecated see {@link OicServerManualConfiguration#getEndSessionUrl()} */
    @Deprecated
    private transient String endSessionEndpoint = null;

    private String postLogoutRedirectUrl;

    @Deprecated
    private transient boolean escapeHatchEnabled = false;

    @Deprecated
    private transient String escapeHatchUsername = null;

    @Deprecated
    private transient Secret escapeHatchSecret = null;

    @Deprecated
    private transient String escapeHatchGroup = null;

    @Deprecated
    /** @deprecated with no replacement.  See sub classes of {@link OicServerConfiguration} */
    private transient String automanualconfigure = null;

    @Deprecated
    /** @deprecated see {@link OicServerWellKnownConfiguration#isUseRefreshTokens()} */
    private transient boolean useRefreshTokens = false;

    private OicServerConfiguration serverConfiguration;

    /** @deprecated with no replacement.  See sub classes of {@link OicServerConfiguration} */
    @Deprecated
    private String overrideScopes = null;

    /** Flag indicating if root url should be taken from config or request
     *
     * Taking root url from request requires a well configured proxy/ingress
     */
    private boolean rootURLFromRequest = false;

    /** Flag to send scopes in code token request
     */
    private boolean sendScopesInTokenRequest = false;

    /**
     * Flag to enable PKCE challenge
     * @deprecated Use {@link Pkce} property instead.
     */
    @Deprecated
    private transient boolean pkceEnabled = false;

    /**
     * Flag to disable JWT signature verification
     * @deprecated Use {@link DisableTokenVerification} property instead.
     */
    @Deprecated
    private transient boolean disableTokenVerification = false;

    /**
     * Flag to disable nonce security
     * @deprecated Use {@link DisableNonce} property instead.
     */
    @Deprecated
    private transient boolean nonceDisabled = false;

    /** Flag to disable token expiration check
     */
    private boolean tokenExpirationCheckDisabled = false;

    /** Flag to enable traditional Jenkins API token based access (no OicSession needed)
     */
    private boolean allowTokenAccessWithoutOicSession = false;

    /**
     * Additional number of seconds to add to token expiration
     * @deprecated Use {@link AllowedTokenExpirationClockSkew} property instead.
     */
    @Deprecated
    private transient Long allowedTokenExpirationClockSkewSeconds = 60L;

    /**
     * Flag when set to true will cause enforce nonce checking in the refresh flow.
     * https://openid.net/specs/openid-connect-core-1_0.html#RefreshTokenResponse states the nonce claim should not be present
     * and when faced with a provider that adheres to this if using a nonce, the library attempts to validate the "missing" nonce and fails.
     * So this is disabled by default, but if the provider does send the nonce in the claim then we do need to verify it.
     * But there is no way to know ahead of time if the server is going to send this or not.
     */
    private static boolean checkNonceInRefreshFlow =
            SystemProperties.getBoolean(OicSecurityRealm.class.getName() + ".checkNonceInRefreshFlow", false);

    /** old field that had an '/' implicitly added at the end,
     * transient because we no longer want to have this value stored,
     * but it's still needed for backwards compatibility */
    @Deprecated
    private transient String endSessionUrl;

    private DescribableList<OidcProperty, OidcPropertyDescriptor> properties = new DescribableList<>(Saveable.NOOP);

    /** Random generator needed for robust random wait
     */
    private static final Random RANDOM = new Random();

    /** Clock used for token expiration check
     */
    private static final Clock CLOCK = Clock.systemUTC();

    /** Runtime context to compile JMESPath
     */
    private static final JmesPath<Object> JMESPATH = new JcfRuntime(
            new RuntimeConfiguration.Builder().withSilentTypeErrors(true).build());

    /**
     * Resource retriever configured with an appropriate SSL Factory based on {@link #isDisableSslVerification()}
     */
    private transient ProxyAwareResourceRetriever proxyAwareResourceRetriever;

    /**
     * @deprecated Use @{link LoginQueryParameters} property instead.
     */
    @Deprecated
    private transient List<LoginQueryParameter> loginQueryParameters;

    /**
     * @deprecated Use @{link LogoutQueryParameters} property instead.
     */
    @Deprecated
    private transient List<LogoutQueryParameter> logoutQueryParameters;

    @DataBoundConstructor
    public OicSecurityRealm(
            String clientId,
            Secret clientSecret,
            OicServerConfiguration serverConfiguration,
            Boolean disableSslVerification,
            IdStrategy userIdStrategy,
            IdStrategy groupIdStrategy)
            throws IOException, FormException {
        // Needed in DataBoundSetter
        this.disableSslVerification = Util.fixNull(disableSslVerification, Boolean.FALSE);
        if (FIPS140.useCompliantAlgorithms() && this.disableSslVerification) {
            throw new FormException(
                    Messages.OicSecurityRealm_DisableSslVerificationFipsMode(), "disableSslVerification");
        }
        this.clientId = clientId;
        this.clientSecret = clientSecret;
        this.serverConfiguration = serverConfiguration;
        this.userIdStrategy = userIdStrategy;
        this.groupIdStrategy = groupIdStrategy;
        this.avatarFieldExpr =
                compileJMESPath("picture", "avatar field"); // Default on OIDC spec, part of profile claim
    }

    @SuppressWarnings("deprecated")
    protected Object readResolve() throws ObjectStreamException {
        if (properties == null) {
            properties = new DescribableList<>(Saveable.NOOP);
        }
        // Fail if migrating to a FIPS non-compliant config
        if (FIPS140.useCompliantAlgorithms() && isDisableSslVerification()) {
            throw new IllegalStateException(Messages.OicSecurityRealm_DisableSslVerificationFipsMode());
        }
        try {
            if (nonceDisabled) {
                properties.replace(new DisableNonce());
            }
            if (pkceEnabled) {
                properties.replace(new Pkce());
            }
            if (disableTokenVerification) {
                properties.replace(new DisableTokenVerification());
            }
            if (allowedTokenExpirationClockSkewSeconds != null) {
                var value = allowedTokenExpirationClockSkewSeconds.intValue();
                if (value != 60) {
                    properties.replace(new AllowedTokenExpirationClockSkew(value));
                }
            }
            if (loginQueryParameters != null) {
                properties.replace(new LoginQueryParameters(loginQueryParameters));
            }
            if (logoutQueryParameters != null) {
                properties.replace(new LogoutQueryParameters(logoutQueryParameters));
            }
            if (escapeHatchEnabled) {
                properties.replace(new EscapeHatch(escapeHatchUsername, escapeHatchGroup, escapeHatchSecret));
            }
        } catch (IOException e) {
            var ose = new InvalidObjectException("Error while migrating properties");
            ose.initCause(e);
            throw ose;
        } catch (FormException e) {
            var ose = new InvalidObjectException(e.getFormField() + ": " + e.getMessage());
            ose.initCause(e);
            throw ose;
        }

        if (!Strings.isNullOrEmpty(endSessionUrl)) {
            this.endSessionEndpoint = endSessionUrl + "/";
        }

        // backward compatibility with wrong groupsFieldName split
        if (Strings.isNullOrEmpty(this.groupsFieldName) && !Strings.isNullOrEmpty(this.simpleGroupsFieldName)) {
            String originalGroupFieldName = this.simpleGroupsFieldName;
            if (!Strings.isNullOrEmpty(this.nestedGroupFieldName)) {
                originalGroupFieldName += "[]." + this.nestedGroupFieldName;
            }
            this.setGroupsFieldName(originalGroupFieldName);
        } else {
            this.setGroupsFieldName(this.groupsFieldName);
        }
        // ensure Field JMESPath are computed
        this.avatarFieldExpr =
                compileJMESPath("picture", "avatar field"); // Default on OIDC spec, part of profile claim
        this.setUserNameField(this.userNameField);
        this.setEmailFieldName(this.emailFieldName);
        this.setFullNameFieldName(this.fullNameFieldName);
        this.setTokenFieldToCheckKey(this.tokenFieldToCheckKey);
        try {
            if (automanualconfigure != null) {
                if ("auto".equals(automanualconfigure)) {
                    OicServerWellKnownConfiguration conf =
                            new OicServerWellKnownConfiguration(wellKnownOpenIDConfigurationUrl);
                    conf.setScopesOverride(this.overrideScopes);
                    serverConfiguration = conf;
                } else {
                    OicServerManualConfiguration conf = new OicServerManualConfiguration(
                            /* TODO */ "migrated", tokenServerUrl, authorizationServerUrl);
                    if (tokenAuthMethod != null) {
                        conf.setTokenAuthMethod(tokenAuthMethod);
                    }
                    conf.setEndSessionUrl(endSessionEndpoint);
                    conf.setJwksServerUrl(jwksServerUrl);
                    conf.setScopes(scopes != null ? scopes : "openid email");
                    conf.setUseRefreshTokens(useRefreshTokens);
                    conf.setUserInfoServerUrl(userInfoServerUrl);
                    serverConfiguration = conf;
                }
            }
        } catch (FormException e) {
            // FormException does not override toString() so looses info on the fields set and the message may not have
            // context
            // extract if into a better message until this is fixed.
            ObjectStreamException ose = new InvalidObjectException(e.getFormField() + ": " + e.getMessage());
            ose.initCause(e);
            throw ose;
        }
        createProxyAwareResourceRetriver();
        return this;
    }

    public String getClientId() {
        return clientId;
    }

    public Secret getClientSecret() {
        return clientSecret == null ? Secret.fromString(NO_SECRET) : clientSecret;
    }

    @Restricted(NoExternalUse.class) // jelly access
    public OicServerConfiguration getServerConfiguration() {
        return serverConfiguration;
    }

    public String getUserNameField() {
        return userNameField;
    }

    @Restricted(NoExternalUse.class)
    public boolean isMissingIdStrategy() {
        return userIdStrategy == null || groupIdStrategy == null;
    }

    @Override
    public IdStrategy getUserIdStrategy() {
        if (userIdStrategy != null) {
            return userIdStrategy;
        } else {
            return IdStrategy.CASE_INSENSITIVE;
        }
    }

    public String getTokenFieldToCheckKey() {
        return tokenFieldToCheckKey;
    }

    public String getTokenFieldToCheckValue() {
        return tokenFieldToCheckValue;
    }

    public String getFullNameFieldName() {
        return fullNameFieldName;
    }

    public String getEmailFieldName() {
        return emailFieldName;
    }

    public String getGroupsFieldName() {
        return groupsFieldName;
    }

    @Override
    public IdStrategy getGroupIdStrategy() {
        if (groupIdStrategy != null) {
            return groupIdStrategy;
        } else {
            return IdStrategy.CASE_INSENSITIVE;
        }
    }

    public boolean isDisableSslVerification() {
        return disableSslVerification;
    }

    public boolean isLogoutFromOpenidProvider() {
        return logoutFromOpenidProvider;
    }

    public String getPostLogoutRedirectUrl() {
        return postLogoutRedirectUrl;
    }

    public boolean isRootURLFromRequest() {
        return rootURLFromRequest;
    }

    public boolean isSendScopesInTokenRequest() {
        return sendScopesInTokenRequest;
    }

    public boolean isTokenExpirationCheckDisabled() {
        return tokenExpirationCheckDisabled;
    }

    public boolean isAllowTokenAccessWithoutOicSession() {
        return allowTokenAccessWithoutOicSession;
    }

    public DescribableList<OidcProperty, OidcPropertyDescriptor> getProperties() {
        return properties;
    }

    @DataBoundSetter
    public void setProperties(List<OidcProperty> properties) throws IOException {
        this.properties.replaceBy(properties);
    }

    @PostConstruct
    @Restricted(NoExternalUse.class)
    public void createProxyAwareResourceRetriver() {
        proxyAwareResourceRetriever =
                ProxyAwareResourceRetriever.createProxyAwareResourceRetriver(isDisableSslVerification());
    }

    ProxyAwareResourceRetriever getResourceRetriever() {
        return proxyAwareResourceRetriever;
    }

    private OidcConfiguration buildOidcConfiguration() {
        // TODO cache this and use the well known if available.
        OidcConfiguration conf = new CustomOidcConfiguration(this.isDisableSslVerification());
        conf.setClientId(clientId);
        conf.setSecret(clientSecret.getPlainText());

        // TODO what do we prefer?
        // conf.setPreferredJwsAlgorithm(JWSAlgorithm.HS256);
        // set many more as needed...

        OIDCProviderMetadata oidcProviderMetadata = serverConfiguration.toProviderMetadata();
        if (oidcProviderMetadata.getScopes() != null) {
            // auto configuration does not need to supply scopes
            conf.setScope(oidcProviderMetadata.getScopes().toString());
        }
        conf.setResourceRetriever(getResourceRetriever());
        return conf;
    }

    @Restricted(NoExternalUse.class) // exposed for testing only
    protected OidcClient buildOidcClient() {
        var executions = properties.stream()
                .map(p -> p.newExecution(serverConfiguration))
                .toList();
        OidcConfiguration oidcConfiguration = buildOidcConfiguration();
        OidcPropertyDescriptor.all().stream()
                .filter(d -> !properties.contains(d))
                .forEach(d -> d.getFallbackConfiguration(serverConfiguration, oidcConfiguration));
        executions.forEach(execution -> execution.customizeConfiguration(oidcConfiguration));
        OidcClient client = new OidcClient(oidcConfiguration);
        // add the extra settings for the client...
        client.setCallbackUrl(buildOAuthRedirectUrl());
        client.setAuthenticator(new OidcAuthenticator(oidcConfiguration, client));
        // when building the redirect URL by default pac4j adds the "client_name=DOidcClient" query parameter to the
        // redirectURL.
        // OPs will reject this for existing clients as the redirect URL is not the same as previously configured
        client.setCallbackUrlResolver(new NoParameterCallbackUrlResolver());
        executions.forEach(execution -> execution.customizeClient(client));
        return client;
    }

    @DataBoundSetter
    public void setUserNameField(String userNameField) {
        this.userNameField = Util.fixNull(Util.fixEmptyAndTrim(userNameField), "sub");
        this.userNameFieldExpr = compileJMESPath(this.userNameField, "user name field");
    }

    @DataBoundSetter
    public void setTokenFieldToCheckKey(String tokenFieldToCheckKey) {
        this.tokenFieldToCheckKey = Util.fixEmptyAndTrim(tokenFieldToCheckKey);
        this.tokenFieldToCheckExpr = compileJMESPath(this.tokenFieldToCheckKey, "token field to check");
    }

    @DataBoundSetter
    public void setTokenFieldToCheckValue(String tokenFieldToCheckValue) {
        this.tokenFieldToCheckValue = Util.fixEmptyAndTrim(tokenFieldToCheckValue);
    }

    @DataBoundSetter
    public void setFullNameFieldName(String fullNameFieldName) {
        this.fullNameFieldName = Util.fixEmptyAndTrim(fullNameFieldName);
        this.fullNameFieldExpr = compileJMESPath(this.fullNameFieldName, "full name field");
    }

    @DataBoundSetter
    public void setEmailFieldName(String emailFieldName) {
        this.emailFieldName = Util.fixEmptyAndTrim(emailFieldName);
        this.emailFieldExpr = compileJMESPath(this.emailFieldName, "email field");
    }

    protected static Expression<Object> compileJMESPath(String str, String logComment) {
        if (str == null) {
            return null;
        }

        try {
            Expression<Object> expr = JMESPATH.compile(str);
            if (expr == null && logComment != null) {
                LOGGER.warning(logComment + " with config '" + str + "' is an invalid JMESPath expression ");
            }
            return expr;
        } catch (RuntimeException e) {
            if (logComment != null) {
                LOGGER.warning(logComment + " config failed " + e);
            }
        }
        return null;
    }

    @DataBoundSetter
    public void setGroupsFieldName(String groupsFieldName) {
        this.groupsFieldName = Util.fixEmptyAndTrim(groupsFieldName);
        this.groupsFieldExpr = compileJMESPath(this.groupsFieldName, "groups field");
    }

    @DataBoundSetter
    public void setLogoutFromOpenidProvider(boolean logoutFromOpenidProvider) {
        this.logoutFromOpenidProvider = logoutFromOpenidProvider;
    }

    @DataBoundSetter
    public void setPostLogoutRedirectUrl(String postLogoutRedirectUrl) {
        this.postLogoutRedirectUrl = Util.fixEmptyAndTrim(postLogoutRedirectUrl);
    }

    @DataBoundSetter
    public void setEscapeHatchSecret(Secret escapeHatchSecret) {
        if (escapeHatchSecret != null) {
            // ensure escapeHatchSecret is BCrypt hash
            String escapeHatchString = Secret.toString(escapeHatchSecret);

            final Pattern BCryptPattern = Pattern.compile("\\A\\$[^$]+\\$\\d+\\$[./0-9A-Za-z]{53}");
            if (!BCryptPattern.matcher(escapeHatchString).matches()) {
                this.escapeHatchSecret = Secret.fromString(BCrypt.hashpw(escapeHatchString, BCrypt.gensalt()));
                return;
            }
        }
        this.escapeHatchSecret = escapeHatchSecret;
    }

    @DataBoundSetter
    public void setRootURLFromRequest(boolean rootURLFromRequest) {
        this.rootURLFromRequest = rootURLFromRequest;
    }

    @DataBoundSetter
    public void setSendScopesInTokenRequest(boolean sendScopesInTokenRequest) {
        this.sendScopesInTokenRequest = sendScopesInTokenRequest;
    }

    @DataBoundSetter
    public void setTokenExpirationCheckDisabled(boolean tokenExpirationCheckDisabled) {
        this.tokenExpirationCheckDisabled = tokenExpirationCheckDisabled;
    }

    @DataBoundSetter
    public void setAllowTokenAccessWithoutOicSession(boolean allowTokenAccessWithoutOicSession) {
        this.allowTokenAccessWithoutOicSession = allowTokenAccessWithoutOicSession;
    }

    @Override
    public String getLoginUrl() {
        // Login begins with our doCommenceLogin(String,String) method
        return "securityRealm/commenceLogin";
    }

    @Override
    public String getAuthenticationGatewayUrl() {
        return "securityRealm/escapeHatch";
    }

    @Override
    public Filter createFilter(FilterConfig filterConfig) {
        return new ChainedServletFilter2(super.createFilter(filterConfig), new Filter() {
            @Override
            public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
                    throws IOException, ServletException {

                if (OicSecurityRealm.this.handleTokenExpiration(
                        (HttpServletRequest) request, (HttpServletResponse) response)) {
                    chain.doFilter(request, response);
                }
            }
        });
    }

    /*
     * Acegi has this notion that first an {@link Authentication} object is created
     * by collecting user information and then the act of authentication is done
     * later (by {@link AuthenticationManager}) to verify it. But in case of OpenID,
     * we create an {@link Authentication} only after we verified the user identity,
     * so {@link AuthenticationManager} becomes no-op.
     */
    @Override
    public SecurityComponents createSecurityComponents() {
        return new SecurityComponents(new AuthenticationManager() {
            public Authentication authenticate(Authentication authentication) throws AuthenticationException {
                if (authentication instanceof AnonymousAuthenticationToken) return authentication;
                for (var property : properties) {
                    var authenticate = property.authenticate(authentication);
                    if (authenticate.isPresent()) {
                        return authenticate.get();
                    }
                }
                throw new BadCredentialsException("Unexpected authentication type: " + authentication);
            }
        });
    }

    /**
     * Validate post-login redirect URL
     *
     * For security reasons, the login must not redirect outside Jenkins
     * realm. For usability reason, the logout page should redirect to
     * root url.
     */
    protected String getValidRedirectUrl(String url) {
        final String rootUrl = getRootUrl();
        if (url != null && !url.isEmpty()) {
            try {
                final String redirectUrl = new URL(new URL(rootUrl), url).toString();
                // check redirect url stays within rootUrl
                if (redirectUrl.startsWith(rootUrl)) {
                    // check if redirect is logout page
                    final String logoutUrl = new URL(new URL(rootUrl), OicLogoutAction.POST_LOGOUT_URL).toString();
                    if (redirectUrl.startsWith(logoutUrl)) {
                        return rootUrl;
                    }
                    return redirectUrl;
                }
            } catch (MalformedURLException e) {
                // Invalid URL, will return root URL
            }
        }
        return rootUrl;
    }

    /**
     * Handles the securityRealm/commenceLogin resource and sends the user off to the IdP
     * @param from the relative URL to the page that the user has just come from
     * @param referer the HTTP referer header (where to redirect the user back to after login has finished)
     * @throws URISyntaxException if the provided data is invalid
     */
    @Restricted(DoNotUse.class) // stapler only
    public void doCommenceLogin(@QueryParameter String from, @Header("Referer") final String referer)
            throws URISyntaxException {

        OidcClient client = buildOidcClient();
        // add the extra params for the client...
        final String redirectOnFinish = getValidRedirectUrl(from != null ? from : referer);

        OidcRedirectionActionBuilder builder = new OidcRedirectionActionBuilder(client);
        FrameworkParameters parameters =
                new JEEFrameworkParameters(Stapler.getCurrentRequest2(), Stapler.getCurrentResponse2());
        WebContext webContext = JEEContextFactory.INSTANCE.newContext(parameters);
        SessionStore sessionStore = JEESessionStoreFactory.INSTANCE.newSessionStore(parameters);
        CallContext ctx = new CallContext(webContext, sessionStore);
        RedirectionAction redirectionAction = builder.getRedirectionAction(ctx).orElseThrow();

        // store the redirect url for after the login.
        sessionStore.set(webContext, SESSION_POST_LOGIN_REDIRECT_URL_KEY, redirectOnFinish);
        JEEHttpActionAdapter.INSTANCE.adapt(redirectionAction, webContext);
    }

    private boolean failedCheckOfTokenField(JWT idToken) throws ParseException {
        if (tokenFieldToCheckKey == null || tokenFieldToCheckValue == null) {
            return false;
        }
        if (idToken == null) {
            return true;
        }
        String value = getStringField(idToken.getJWTClaimsSet().getClaims(), tokenFieldToCheckExpr);
        if (value == null) {
            return true;
        }
        return !tokenFieldToCheckValue.equals(value);
    }

    private void loginAndSetUserData(
            String userName, JWT idToken, Map<String, Object> userInfo, OicCredentials credentials)
            throws IOException, ParseException {

        List<GrantedAuthority> grantedAuthorities = determineAuthorities(idToken, userInfo);
        if (LOGGER.isLoggable(Level.FINEST)) {
            StringBuilder grantedAuthoritiesAsString = new StringBuilder(userName);
            grantedAuthoritiesAsString.append(" (");
            for (GrantedAuthority grantedAuthority : grantedAuthorities) {
                grantedAuthoritiesAsString.append(" ").append(grantedAuthority.getAuthority());
            }
            grantedAuthoritiesAsString.append(" )");
            LOGGER.finest("GrantedAuthorities:" + grantedAuthoritiesAsString);
        }

        UsernamePasswordAuthenticationToken token =
                new UsernamePasswordAuthenticationToken(userName, "", grantedAuthorities);

        SecurityContextHolder.getContext().setAuthentication(token);

        User user = User.get2(token);
        if (user == null) {
            // should not happen
            throw new IOException("Cannot set OIDC property on anonymous user");
        }
        String email = determineStringField(emailFieldExpr, idToken, userInfo);
        if (email != null) {
            user.addProperty(new Mailer.UserProperty(email));
        }

        String fullName = determineStringField(fullNameFieldExpr, idToken, userInfo);
        if (fullName != null) {
            user.setFullName(fullName);
        }

        // Set avatar if possible
        String avatarUrl = determineStringField(avatarFieldExpr, idToken, userInfo);
        OicAvatarProperty oicAvatarProperty;
        if (avatarUrl != null) {
            LOGGER.finest(() -> "Avatar url is: " + avatarUrl);
            OicAvatarProperty.AvatarImage avatarImage = new OicAvatarProperty.AvatarImage(avatarUrl);
            oicAvatarProperty = new OicAvatarProperty(avatarImage);
        } else {
            LOGGER.finest(() -> "No avatar URL found for user " + user.getId() + ". Ensure to remove existing avatar");
            oicAvatarProperty = new OicAvatarProperty(null);
        }
        user.addProperty(oicAvatarProperty);

        user.addProperty(credentials);

        OicUserDetails userDetails = new OicUserDetails(userName, grantedAuthorities);
        SecurityListener.fireAuthenticated2(userDetails);
        SecurityListener.fireLoggedIn(userName);
    }

    private String determineStringField(Expression<Object> fieldExpr, JWT idToken, Map<String, Object> userInfo)
            throws ParseException {
        if (fieldExpr != null) {
            if (userInfo != null) {
                Object field = fieldExpr.search(userInfo);
                if (field != null) {
                    if (field instanceof String) {
                        String fieldValue = Util.fixEmptyAndTrim((String) field);
                        if (fieldValue != null) {
                            return fieldValue;
                        }
                    }
                    // pac4j OIDC client returns URI for some fields like the "picture" field
                    if (field instanceof URI) {
                        return ((URI) field).toASCIIString();
                    }
                }
            }
            if (idToken != null) {
                String fieldValue = Util.fixEmptyAndTrim(
                        getStringField(idToken.getJWTClaimsSet().getClaims(), fieldExpr));
                if (fieldValue != null) {
                    return fieldValue;
                }
            }
        }
        return null;
    }

    protected String getStringField(Object object, Expression<Object> fieldExpr) {
        if (object != null && fieldExpr != null) {
            Object value = fieldExpr.search(object);
            if ((value != null) && !(value instanceof Map) && !(value instanceof List)) {
                return String.valueOf(value);
            }
        }
        return null;
    }

    private List<GrantedAuthority> determineAuthorities(JWT idToken, Map<String, Object> userInfo)
            throws ParseException {
        List<GrantedAuthority> grantedAuthorities = new ArrayList<>();
        grantedAuthorities.add(SecurityRealm.AUTHENTICATED_AUTHORITY2);
        if (this.groupsFieldExpr == null) {
            if (this.groupsFieldName == null) {
                LOGGER.fine("Not adding groups because groupsFieldName is not set.");
            } else {
                LOGGER.fine("Not adding groups because groupsFieldName is invalid. groupsFieldName=" + groupsFieldName);
            }
            return grantedAuthorities;
        }

        Object groupsObject = null;

        // userInfo has precedence when available
        if (userInfo != null) {
            groupsObject = this.groupsFieldExpr.search(userInfo);
        }
        if (groupsObject == null && idToken != null) {
            groupsObject = this.groupsFieldExpr.search(idToken.getJWTClaimsSet().getClaims());
        }
        if (groupsObject == null) {
            LOGGER.warning("idToken and userInfo did not contain group field name: " + this.groupsFieldName);
            return grantedAuthorities;
        }

        List<String> groupNames = ensureString(groupsObject);
        if (groupNames.isEmpty()) {
            LOGGER.warning("Could not identify groups in " + groupsFieldName + "=" + groupsObject);
            return grantedAuthorities;
        }
        LOGGER.fine("Number of groups in groupNames: " + groupNames.size());

        for (String groupName : groupNames) {
            LOGGER.fine("Adding group from UserInfo: " + groupName);
            grantedAuthorities.add(new SimpleGrantedAuthority(groupName));
        }

        return grantedAuthorities;
    }

    /** Ensure group field object returns is string or list of string
     */
    private List<String> ensureString(Object field) {
        if (field == null) {
            LOGGER.warning("userInfo did not contain a valid group field content, got null");
            return Collections.emptyList();
        } else if (field instanceof String sField) {
            // if it's a String, the original value was not a json array.
            // We try to convert the string to list based on comma while ignoring whitespaces and square brackets.
            // Example value "[demo-user-group, demo-test-group, demo-admin-group]"
            String[] rawFields = sField.split("[\\s\\[\\],]");
            List<String> result = new ArrayList<>();
            for (String rawField : rawFields) {
                if (rawField != null && !rawField.isEmpty()) {
                    result.add(rawField);
                }
            }
            return result;
        } else if (field instanceof List) {
            List<String> result = new ArrayList<>();
            List<Object> groups = (List<Object>) field;
            for (Object group : groups) {
                if (group instanceof String) {
                    result.add(group.toString());
                } else if (group instanceof Map) {
                    // if it's a Map, we use the nestedGroupFieldName to grab the groups
                    Map<String, String> groupMap = (Map<String, String>) group;
                    if (nestedGroupFieldName != null && groupMap.containsKey(nestedGroupFieldName)) {
                        result.add(groupMap.get(nestedGroupFieldName));
                    }
                }
            }
            return result;
        } else {
            try {
                return (List<String>) field;
            } catch (ClassCastException e) {
                LOGGER.warning("userInfo did not contain a valid group field content, got: "
                        + field.getClass().getSimpleName());
                return Collections.emptyList();
            }
        }
    }

    @Restricted(DoNotUse.class) // stapler only
    public void doLogout(StaplerRequest2 req, StaplerResponse2 rsp) throws IOException, ServletException {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        User user = User.get2(authentication);

        Assert.notNull(user, "User must not be null");

        OicCredentials credentials = user.getProperty(OicCredentials.class);

        if (credentials != null) {
            if (this.logoutFromOpenidProvider
                    && serverConfiguration.toProviderMetadata().getEndSessionEndpointURI() != null) {
                // This ensures that token will be expired at the right time with API Key calls, but no refresh can be
                // made.
                user.addProperty(new OicCredentials(null, null, null, CLOCK.millis()));
            }

            req.setAttribute(ID_TOKEN_REQUEST_ATTRIBUTE, credentials.getIdToken());
        }

        super.doLogout(req, rsp);
    }

    @Override
    public String getPostLogOutUrl2(StaplerRequest2 req, Authentication auth) {
        Object idToken = req.getAttribute(ID_TOKEN_REQUEST_ATTRIBUTE);
        Object state = getStateAttribute();
        var openidLogoutEndpoint = maybeOpenIdLogoutEndpoint(
                Objects.toString(idToken, ""), Objects.toString(state), this.postLogoutRedirectUrl);
        if (openidLogoutEndpoint != null) {
            return openidLogoutEndpoint;
        }
        return getFinalLogoutUrl(req, auth);
    }

    @VisibleForTesting
    Object getStateAttribute() {
        // return null;
        OidcClient client = buildOidcClient();
        FrameworkParameters parameters =
                new JEEFrameworkParameters(Stapler.getCurrentRequest2(), Stapler.getCurrentResponse2());
        WebContext webContext = JEEContextFactory.INSTANCE.newContext(parameters);
        SessionStore sessionStore = JEESessionStoreFactory.INSTANCE.newSessionStore(parameters);
        CallContext ctx = new CallContext(webContext, sessionStore);
        return client.getConfiguration()
                .getValueRetriever()
                .retrieve(ctx, client.getStateSessionAttributeName(), client)
                .orElse(null);
    }

    @CheckForNull
    String maybeOpenIdLogoutEndpoint(String idToken, String state, String postLogoutRedirectUrl) {
        final URI url = serverConfiguration.toProviderMetadata().getEndSessionEndpointURI();
        if (this.logoutFromOpenidProvider && url != null) {
            var uriComponentsBuilder = UriComponentsBuilder.fromUri(url);
            if (!Strings.isNullOrEmpty(idToken)) {
                uriComponentsBuilder.queryParam("id_token_hint", idToken);
            }
            if (!Strings.isNullOrEmpty(state) && !"null".equals(state)) {
                uriComponentsBuilder.queryParam("state", state);
            }
            if (postLogoutRedirectUrl != null) {
                uriComponentsBuilder.queryParam(
                        "post_logout_redirect_uri", URLEncoder.encode(postLogoutRedirectUrl, StandardCharsets.UTF_8));
            }
            properties.stream()
                    .flatMap(p -> p.contributeLogoutQueryParameters().stream())
                    .forEach(lqp -> {
                        var urlEncodedValue = lqp.getURLEncodedValue();
                        uriComponentsBuilder.queryParam(
                                lqp.getURLEncodedKey(), urlEncodedValue.isEmpty() ? null : urlEncodedValue);
                    });
            return uriComponentsBuilder.build().toUriString();
        }
        return null;
    }

    private String getFinalLogoutUrl(StaplerRequest2 req, Authentication auth) {
        if (Jenkins.get().hasPermission(Jenkins.READ)) {
            return super.getPostLogOutUrl2(req, auth);
        }
        return req.getContextPath() + "/" + OicLogoutAction.POST_LOGOUT_URL;
    }

    private String getRootUrl() {
        if (rootURLFromRequest) {
            return Jenkins.get().getRootUrlFromRequest();
        } else {
            return Jenkins.get().getRootUrl();
        }
    }

    private String ensureRootUrl() {
        String rootUrl = getRootUrl();
        if (rootUrl == null) {
            throw new NullPointerException("Jenkins root url must not be null");
        } else {
            return rootUrl;
        }
    }

    private String buildOAuthRedirectUrl() throws NullPointerException {
        return ensureRootUrl() + "securityRealm/finishLogin";
    }

    /**
     * This is where the user comes back to at the end of the OpenID redirect ping-pong.
     * @param request The user's request
     * @throws ParseException if the JWT (or other response) could not be parsed.
     */
    public void doFinishLogin(StaplerRequest2 request, StaplerResponse2 response) throws IOException, ParseException {
        OidcClient client = buildOidcClient();

        FrameworkParameters parameters = new JEEFrameworkParameters(request, response);
        WebContext webContext = JEEContextFactory.INSTANCE.newContext(parameters);
        SessionStore sessionStore = JEESessionStoreFactory.INSTANCE.newSessionStore(parameters);

        try {
            // NB: TODO this also handles back channel logout if "logoutendpoint" parameter is set
            // see  org.pac4j.oidc.credentials.extractor.OidcExtractor.extract(WebContext, SessionStore)
            // but we probably need to hookup a special LogoutHandler in the clients configuration to do all the special
            // Jenkins stuff correctly
            // also should have its own URL to make the code easier to follow :)

            if (!sessionStore.renewSession(webContext)) {
                throw new TechnicalException("Could not create a new session");
            }

            CallContext ctx = new CallContext(webContext, sessionStore);
            Credentials credentials = client.getCredentials(ctx)
                    .orElseThrow(() -> new Failure("Could not extract credentials from request"));
            credentials = client.validateCredentials(ctx, credentials)
                    .orElseThrow(() -> new Failure("Could not validate credentials from request"));

            ProfileCreator profileCreator = client.getProfileCreator();

            // creating the profile performs validation of the token
            OidcProfile profile = (OidcProfile) profileCreator
                    .create(ctx, credentials)
                    .orElseThrow(() -> new Failure("Could not build user profile"));

            AccessToken accessToken = profile.getAccessToken();
            JWT idToken = profile.getIdToken();
            RefreshToken refreshToken = profile.getRefreshToken();

            String username = determineStringField(userNameFieldExpr, idToken, profile.getAttributes());
            if (failedCheckOfTokenField(idToken)) {
                throw new FailedCheckOfTokenException(client.getConfiguration().findLogoutUrl());
            }

            OicCredentials oicCredentials = new OicCredentials(
                    accessToken == null ? null : accessToken.getValue(), // XXX (how) can the access token be null?
                    idToken.getParsedString(),
                    refreshToken != null ? refreshToken.getValue() : null,
                    accessToken == null ? 0 : accessToken.getLifetime(),
                    CLOCK.millis(),
                    (long) client.getConfiguration().getMaxClockSkew());

            loginAndSetUserData(username, idToken, profile.getAttributes(), oicCredentials);

            String redirectUrl = (String) sessionStore
                    .get(webContext, SESSION_POST_LOGIN_REDIRECT_URL_KEY)
                    .orElse(Jenkins.get().getRootUrl());
            if (redirectUrl != null) {
                response.sendRedirect(HttpURLConnection.HTTP_MOVED_TEMP, redirectUrl);
            } else {
                response.sendError(HttpURLConnection.HTTP_INTERNAL_ERROR, "redirectUrl was null for the current flow");
            }

        } catch (HttpAction e) {
            // this may be an OK flow for logout login is handled upstream.
            JEEHttpActionAdapter.INSTANCE.adapt(e, webContext);
        }
    }

    /**
     * Handles Token Expiration.
     * @throws IOException a low level exception
     */
    public boolean handleTokenExpiration(HttpServletRequest httpRequest, HttpServletResponse httpResponse)
            throws IOException {
        if (isLogoutRequest(httpRequest)) {
            // No need to refresh token when logging out
            return true;
        }

        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        User user = User.get2(authentication);
        if (user == null) {
            return true;
        }

        OicCredentials credentials = user.getProperty(OicCredentials.class);

        if (credentials == null) {
            return true;
        }

        if (isValidApiTokenRequest(httpRequest, user)) {
            return true;
        }

        if (isExpired(credentials)) {
            if (canRefreshToken(credentials)) {
                LOGGER.log(Level.FINEST, "Attempting to refresh credential for user: {0}", user.getId());
                boolean retVal = refreshExpiredToken(user.getId(), credentials, httpRequest, httpResponse);
                LOGGER.log(Level.FINEST, "Refresh credential for user returned {0}", retVal);
                return retVal;
            } else if (!isTokenExpirationCheckDisabled()) {
                redirectToLoginUrl(httpRequest, httpResponse);
                return false;
            }
        }

        return true;
    }

    boolean isLogoutRequest(HttpServletRequest request) {
        return request.getRequestURI().endsWith("/logout");
    }

    boolean isValidApiTokenRequest(HttpServletRequest httpRequest, User user) {
        if (isAllowTokenAccessWithoutOicSession()) {
            // check if this is a valid api token based request
            String authHeader = httpRequest.getHeader("Authorization");
            if (authHeader != null && authHeader.startsWith("Basic ")) {
                String token = new String(Base64.getDecoder().decode(authHeader.substring(6)), StandardCharsets.UTF_8)
                        .split(":")[1];

                // this was a valid jenkins token being used, exit this filter and let
                // the rest of chain be processed
                // else do nothing and continue evaluating this request
                ApiTokenProperty apiTokenProperty = user.getProperty(ApiTokenProperty.class);
                return apiTokenProperty != null && apiTokenProperty.matchesPassword(token);
            }
        }
        return false;
    }

    boolean canRefreshToken(OicCredentials credentials) {
        return serverConfiguration.toProviderMetadata().getGrantTypes() != null
                && serverConfiguration.toProviderMetadata().getGrantTypes().contains(GrantType.REFRESH_TOKEN)
                && !Strings.isNullOrEmpty(credentials.getRefreshToken());
    }

    private void redirectToLoginUrl(HttpServletRequest req, HttpServletResponse res) throws IOException {
        if (req != null && (req.getSession(false) != null || Strings.isNullOrEmpty(req.getHeader("Authorization")))) {
            req.getSession().invalidate();
        }
        if (res != null) {
            res.sendRedirect(Jenkins.get().getSecurityRealm().getLoginUrl());
        }
    }

    public boolean isExpired(OicCredentials credentials) {
        if (credentials.getExpiresAtMillis() == null) {
            return false;
        }

        return CLOCK.millis() >= credentials.getExpiresAtMillis();
    }

    private boolean refreshExpiredToken(
            String expectedUsername,
            OicCredentials credentials,
            HttpServletRequest httpRequest,
            HttpServletResponse httpResponse)
            throws IOException {

        FrameworkParameters parameters = new JEEFrameworkParameters(httpRequest, httpResponse);
        WebContext webContext = JEEContextFactory.INSTANCE.newContext(parameters);
        SessionStore sessionStore = JEESessionStoreFactory.INSTANCE.newSessionStore(parameters);
        OidcClient client = buildOidcClient();
        // PAC4J maintains the nonce even though servers should not respond with an id token containing the nonce
        // https://openid.net/specs/openid-connect-core-1_0.html#RefreshTokenResponse
        // it SHOULD NOT have a nonce Claim, even when the ID Token issued at the time of the original authentication
        // contained nonce;
        // however, if it is present, its value MUST be the same as in the ID Token issued at the time of the original
        // authentication
        // by default we will strip out the nonce unless the user has opted into it.
        var configuration = client.getConfiguration();
        configuration.setUseNonce(configuration.isUseNonce() && checkNonceInRefreshFlow);
        try {
            OidcProfile profile = new OidcProfile();
            profile.setAccessToken(new BearerAccessToken(credentials.getAccessToken()));
            profile.setIdTokenString(credentials.getIdToken());
            profile.setRefreshToken(new RefreshToken(credentials.getRefreshToken()));

            CallContext ctx = new CallContext(webContext, sessionStore);
            profile = (OidcProfile) client.renewUserProfile(ctx, profile)
                    .orElseThrow(() -> new IllegalStateException("Could not renew user profile"));

            // During refresh the IDToken may or may not be present.
            // The refresh token may also not be present.
            // in these cases we will reuse the original values.

            AccessToken accessToken = profile.getAccessToken();
            JWT idToken = Objects.requireNonNullElse(profile.getIdToken(), JWTParser.parse(credentials.getIdToken()));
            RefreshToken refreshToken = Objects.requireNonNullElse(
                    profile.getRefreshToken(), new RefreshToken(credentials.getRefreshToken()));

            String username = determineStringField(userNameFieldExpr, idToken, profile.getAttributes());
            if (!User.idStrategy().equals(expectedUsername, username)) {
                httpResponse.sendError(
                        HttpServletResponse.SC_UNAUTHORIZED, "User name was not the same after refresh request");
                return false;
            }
            // the username may have changed case during a call, but still be the same user (as we have checked the
            // idStrategy)
            // we need to keep using exactly the same principal otherwise there is a potential for crumbs not to match.
            // whilst we could do some normalization of the username, just use the original (expected) username
            // see https://github.com/jenkinsci/oic-auth-plugin/issues/411
            if (LOGGER.isLoggable(Level.FINE)) {
                Authentication a = SecurityContextHolder.getContext().getAuthentication();
                User u = User.get2(a);
                LOGGER.log(
                        Level.FINE,
                        "Token refresh.  Current Authentication principal: " + a.getName() + " user id:"
                                + (u == null ? "null user" : u.getId()) + " newly retrieved username would have been: "
                                + username);
            }
            username = expectedUsername;

            if (failedCheckOfTokenField(idToken)) {
                throw new FailedCheckOfTokenException(configuration.findLogoutUrl());
            }

            OicCredentials refreshedCredentials = new OicCredentials(
                    accessToken.getValue(),
                    idToken.getParsedString(),
                    refreshToken.getValue(),
                    accessToken.getLifetime(),
                    CLOCK.millis(),
                    (long) client.getConfiguration().getMaxClockSkew());

            loginAndSetUserData(username, idToken, profile.getAttributes(), refreshedCredentials);
            return true;
        } catch (TechnicalException e) {
            if (StringUtils.contains(e.getMessage(), "error=invalid_grant")) {
                // the code is lost from the TechnicalException so we need to resort to string matching
                // to retain the same flow :-(
                if (isTokenExpirationCheckDisabled()) {
                    // the code is lost from the TechnicalException so we need to resort to string matching to retain
                    // the same flow :-(
                    LOGGER.log(
                            Level.FINE,
                            "Failed to refresh expired token because grant is invalid, proceeding as \"Token Expiration Check Disabled\" is set");
                    return false;
                }
                LOGGER.log(Level.FINE, "Failed to refresh expired token", e);
                redirectToLoginUrl(httpRequest, httpResponse);
                return false;
            }
            LOGGER.log(Level.WARNING, "Failed to refresh expired token", e);
            redirectToLoginUrl(httpRequest, httpResponse);
            return false;
        } catch (ParseException e) {
            LOGGER.log(Level.WARNING, "Failed to refresh expired token", e);
            // could not renew
            redirectToLoginUrl(httpRequest, httpResponse);
            return false;
        } catch (IllegalStateException e) {
            LOGGER.log(Level.WARNING, "Failed to refresh expired token, profile was null", e);
            // could not renew
            redirectToLoginUrl(httpRequest, httpResponse);
            return false;
        }
    }

    @Extension
    public static final class DescriptorImpl extends Descriptor<SecurityRealm> {

        @Override
        public String getDisplayName() {
            return Messages.OicSecurityRealm_DisplayName();
        }

        @RequirePOST
        public FormValidation doCheckClientId(@QueryParameter String clientId) {
            Jenkins.get().checkPermission(Jenkins.ADMINISTER);
            if (Util.fixEmptyAndTrim(clientId) == null) {
                return FormValidation.error(Messages.OicSecurityRealm_ClientIdRequired());
            }
            return FormValidation.ok();
        }

        @RequirePOST
        public FormValidation doCheckClientSecret(@QueryParameter String clientSecret) {
            Jenkins.get().checkPermission(Jenkins.ADMINISTER);
            if (Util.fixEmptyAndTrim(clientSecret) == null) {
                return FormValidation.error(Messages.OicSecurityRealm_ClientSecretRequired());
            }
            return FormValidation.ok();
        }

        @RequirePOST
        public FormValidation doCheckPostLogoutRedirectUrl(@QueryParameter String postLogoutRedirectUrl) {
            Jenkins.get().checkPermission(Jenkins.ADMINISTER);
            if (Util.fixEmptyAndTrim(postLogoutRedirectUrl) != null) {
                try {
                    new URL(postLogoutRedirectUrl);
                    return FormValidation.ok();
                } catch (MalformedURLException e) {
                    return FormValidation.error(e, Messages.OicSecurityRealm_NotAValidURL());
                }
            }

            return FormValidation.ok();
        }

        @RequirePOST
        public FormValidation doCheckUserNameField(@QueryParameter String userNameField) {
            return this.doCheckFieldName(
                    userNameField, FormValidation.ok(Messages.OicSecurityRealm_UsingDefaultUsername()));
        }

        @RequirePOST
        public FormValidation doCheckFullNameFieldName(@QueryParameter String fullNameFieldName) {
            return this.doCheckFieldName(fullNameFieldName, FormValidation.ok());
        }

        @RequirePOST
        public FormValidation doCheckEmailFieldName(@QueryParameter String emailFieldName) {
            return this.doCheckFieldName(emailFieldName, FormValidation.ok());
        }

        @RequirePOST
        public FormValidation doCheckGroupsFieldName(@QueryParameter String groupsFieldName) {
            return this.doCheckFieldName(groupsFieldName, FormValidation.ok());
        }

        @RequirePOST
        public FormValidation doCheckTokenFieldToCheckKey(@QueryParameter String tokenFieldToCheckKey) {
            return this.doCheckFieldName(tokenFieldToCheckKey, FormValidation.ok());
        }

        @RequirePOST
        public FormValidation doCheckDisableSslVerification(@QueryParameter Boolean disableSslVerification) {
            if (FIPS140.useCompliantAlgorithms() && disableSslVerification) {
                return FormValidation.error(Messages.OicSecurityRealm_DisableSslVerificationFipsMode());
            }
            return FormValidation.ok();
        }

        // method to check fieldName matches JMESPath format
        private FormValidation doCheckFieldName(String fieldName, FormValidation validIfNull) {
            Jenkins.get().checkPermission(Jenkins.ADMINISTER);
            if (Util.fixEmptyAndTrim(fieldName) == null) {
                return validIfNull;
            }
            if (OicSecurityRealm.compileJMESPath(fieldName, null) == null) {
                return FormValidation.error(Messages.OicSecurityRealm_InvalidFieldName());
            }
            return FormValidation.ok();
        }

        @Restricted(NoExternalUse.class) // jelly only
        public Descriptor<OicServerConfiguration> getDefaultServerConfigurationType() {
            return Jenkins.get().getDescriptor(OicServerWellKnownConfiguration.class);
        }

        @Restricted(NoExternalUse.class) // used by jelly only
        public boolean isFipsEnabled() {
            return FIPS140.useCompliantAlgorithms();
        }

        @Restricted(NoExternalUse.class)
        public List<IdStrategyDescriptor> getIdStrategyDescriptors() {
            return ExtensionList.lookup(IdStrategyDescriptor.class);
        }

        /**
         * The default username strategy for new OicSecurityRealm
         */
        @Restricted(NoExternalUse.class)
        public IdStrategy defaultUsernameIdStrategy() {
            return new IdStrategy.CaseSensitive();
        }

        /**
         * The default group strategy for new OicSecurityRealm
         */
        @Restricted(NoExternalUse.class)
        public IdStrategy defaultGroupIdStrategy() {
            return new IdStrategy.CaseSensitive();
        }

        @SuppressWarnings("unused") // stapler
        public List<OidcPropertyDescriptor> getPropertiesDescriptors() {
            return ExtensionList.lookup(OidcPropertyDescriptor.class).stream()
                    .filter(OidcPropertyDescriptor::isApplicable)
                    .toList();
        }
    }
}
