package org.jenkinsci.plugins.oic.properties;

import static org.apache.commons.lang.StringUtils.isNotBlank;

import edu.umd.cs.findbugs.annotations.CheckForNull;
import edu.umd.cs.findbugs.annotations.NonNull;
import hudson.Extension;
import hudson.Util;
import hudson.model.Descriptor;
import hudson.security.SecurityRealm;
import hudson.security.csrf.CrumbExclusion;
import hudson.util.Secret;
import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.Serial;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.regex.Pattern;
import jenkins.security.FIPS140;
import jenkins.security.SecurityListener;
import org.jenkinsci.plugins.oic.Messages;
import org.jenkinsci.plugins.oic.OicUserDetails;
import org.jenkinsci.plugins.oic.OidcProperty;
import org.jenkinsci.plugins.oic.OidcPropertyDescriptor;
import org.kohsuke.stapler.DataBoundConstructor;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.crypto.bcrypt.BCrypt;

/**
 * Escape hatch for authentication, allowing users to log in with a username and password.
 * This is intended for emergency access and should be used with caution.
 */
public class EscapeHatch extends OidcProperty {
    public static final Pattern B_CRYPT_PATTERN = Pattern.compile("\\A\\$[^$]+\\$\\d+\\$[./0-9A-Za-z]{53}");

    @NonNull
    private final String username;

    @CheckForNull
    private final String group;

    @NonNull
    private final Secret secret;

    @DataBoundConstructor
    public EscapeHatch(@NonNull String username, @CheckForNull String group, @NonNull Secret secret)
            throws Descriptor.FormException {
        if (FIPS140.useCompliantAlgorithms()) {
            throw new Descriptor.FormException("Cannot use Escape Hatch in FIPS-140 mode", "escapeHatch");
        }
        var sanitizedUsername = Util.fixEmptyAndTrim(username);
        if (sanitizedUsername == null) {
            throw new Descriptor.FormException("Username must not be blank", "username");
        }
        this.username = sanitizedUsername;
        this.group = Util.fixEmptyAndTrim(group);
        // ensure secret is BCrypt hash
        String secretAsString = Secret.toString(secret);
        if (B_CRYPT_PATTERN.matcher(secretAsString).matches()) {
            this.secret = secret;
        } else {
            this.secret = Secret.fromString(BCrypt.hashpw(secretAsString, BCrypt.gensalt()));
        }
    }

    @CheckForNull
    public String getUsername() {
        return username;
    }

    @CheckForNull
    public String getGroup() {
        return group;
    }

    @NonNull
    public Secret getSecret() {
        return secret;
    }

    /**
     * Random generator needed for robust random wait
     */
    private static final Random RANDOM = new Random();

    private void randomWait() {
        try {
            Thread.sleep(1000L + RANDOM.nextLong(1000L));
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    @NonNull
    @Override
    public Optional<Authentication> authenticate(@NonNull Authentication authentication) {
        if (authentication instanceof UsernamePasswordAuthenticationToken) {
            randomWait(); // to slowdown brute forcing
            if (check(
                    authentication.getPrincipal().toString(),
                    authentication.getCredentials().toString())) {
                List<GrantedAuthority> grantedAuthorities = new ArrayList<>();
                grantedAuthorities.add(SecurityRealm.AUTHENTICATED_AUTHORITY2);
                if (isNotBlank(group)) {
                    grantedAuthorities.add(new SimpleGrantedAuthority(group));
                }
                UsernamePasswordAuthenticationToken token =
                        new UsernamePasswordAuthenticationToken(username, "", grantedAuthorities);
                SecurityContextHolder.getContext().setAuthentication(token);
                OicUserDetails userDetails = new OicUserDetails(username, grantedAuthorities);
                SecurityListener.fireAuthenticated2(userDetails);
                return Optional.of(token);
            } else {
                throw new BadCredentialsException("Wrong username and password: " + authentication);
            }
        }
        return Optional.empty();
    }

    /**
     * Check a given username and password against the configured ones.
     */
    public boolean check(@NonNull String username, @CheckForNull String password) {
        return username.equals(this.username) && BCrypt.checkpw(password, Secret.toString(this.secret));
    }

    public static class DescriptorImpl extends OidcPropertyDescriptor {
        @Extension
        @CheckForNull
        public static DescriptorImpl createIfApplicable() {
            if (FIPS140.useCompliantAlgorithms()) {
                return null;
            }
            return new DescriptorImpl();
        }

        @NonNull
        @Override
        public String getDisplayName() {
            return "Escape Hatch";
        }
    }

    @Serial
    protected Object readResolve() {
        if (FIPS140.useCompliantAlgorithms()) {
            throw new IllegalStateException(Messages.OicSecurityRealm_EscapeHatchFipsMode());
        }
        return this;
    }

    /**
     * Excluding the escapeHatch login from CSRF protection as the crumb is calculated based on the authentication
     * mirroring behavior of the normal login page.
     *
     * @author Michael Bischoff
     */
    @Extension
    public static class CrumbExclusionImpl extends CrumbExclusion {

        @Override
        public boolean process(HttpServletRequest request, HttpServletResponse response, FilterChain chain)
                throws IOException, ServletException {
            String pathInfo = request.getPathInfo();
            if ("/securityRealm/escapeHatch".equals(pathInfo)) {
                chain.doFilter(request, response);
                return true;
            }
            return false;
        }
    }
}
