/*
 * Decompiled with CFR 0.152.
 */
package hudson.plugins.libvirt;

import hudson.model.Computer;
import hudson.model.TaskListener;
import hudson.plugins.libvirt.VirtualMachine;
import hudson.plugins.libvirt.lib.IDomain;
import hudson.plugins.libvirt.lib.IDomainSnapshot;
import hudson.plugins.libvirt.lib.VirtException;
import hudson.remoting.VirtualChannel;
import hudson.slaves.OfflineCause;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.CheckForNull;

public final class ComputerUtils {
    private static final Logger LOGGER = Logger.getLogger(ComputerUtils.class.getName());
    private static final int RETRY_MAX = 5;
    private static final int RETRY_WAIT_MS = 500;

    private ComputerUtils() {
    }

    public static void disconnect(String name, Computer computer, OfflineCause cause) {
        ComputerUtils.disconnect(name, computer, null, cause);
    }

    public static void disconnect(String name, Computer computer, @CheckForNull TaskListener listener, OfflineCause cause) {
        VirtualChannel virtualChannel = computer.getChannel();
        if (virtualChannel == null) {
            ComputerUtils.error(listener, "Could not determine channel.");
            return;
        }
        try {
            virtualChannel.syncLocalIO();
            try {
                virtualChannel.close();
                computer.disconnect(cause);
                try {
                    computer.waitUntilOffline();
                }
                catch (InterruptedException e) {
                    ComputerUtils.error(listener, "Interrupted while waiting for computer to be offline: " + String.valueOf(e));
                }
            }
            catch (IOException e) {
                ComputerUtils.error(listener, "Error closing channel: " + String.valueOf(e));
            }
        }
        catch (InterruptedException e) {
            ComputerUtils.error(listener, "Interrupted while syncing IO: " + String.valueOf(e));
        }
    }

    public static void start(VirtualMachine virtualMachine) {
        ComputerUtils.start(virtualMachine, null);
    }

    public static void start(VirtualMachine virtualMachine, @CheckForNull TaskListener listener) {
        IDomain domain = ComputerUtils.getDomain(virtualMachine, listener);
        if (domain != null) {
            try {
                if (domain.isRunningOrBlocked()) {
                    ComputerUtils.log(listener, "VM " + virtualMachine.getName() + " is already running, no startup required.");
                    return;
                }
            }
            catch (VirtException e) {
                ComputerUtils.error(listener, "Error checking if VM " + virtualMachine.getName() + " is already running, will consider it as stopped.");
            }
            for (int i = 0; i < 5; ++i) {
                try {
                    ComputerUtils.log(listener, "Starting VM \"" + virtualMachine.getName() + "\"");
                    domain.create();
                    break;
                }
                catch (VirtException e) {
                    try {
                        Thread.sleep(500L);
                    }
                    catch (InterruptedException interruptedException) {
                        // empty catch block
                    }
                    continue;
                }
            }
        }
    }

    public static void stop(VirtualMachine virtualMachine, String shutdownMethod) {
        ComputerUtils.stop(virtualMachine, shutdownMethod, null);
    }

    public static void stop(VirtualMachine virtualMachine, String shutdownMethod, @CheckForNull TaskListener listener) {
        IDomain domain = ComputerUtils.getDomain(virtualMachine, listener);
        if (domain != null) {
            try {
                if (domain.isNotBlockedAndNotRunning()) {
                    ComputerUtils.log(listener, "VM \"" + virtualMachine.getName() + "\" is not running, no shutdown required.");
                    return;
                }
            }
            catch (VirtException e) {
                ComputerUtils.error(listener, MessageFormat.format("Error checking if VM \"{0}\" is stopped, will consider it as running.", virtualMachine.getName()));
            }
            for (int i = 0; i < 5; ++i) {
                try {
                    ComputerUtils.log(listener, MessageFormat.format("Stopping VM \"{0}\" (using method {1})", virtualMachine.getName(), shutdownMethod));
                    if ("suspend".equals(shutdownMethod)) {
                        domain.suspend();
                        break;
                    }
                    if ("destroy".equals(shutdownMethod)) {
                        domain.destroy();
                        break;
                    }
                    domain.shutdown();
                    break;
                }
                catch (VirtException e) {
                    try {
                        Thread.sleep(500L);
                    }
                    catch (InterruptedException interruptedException) {
                        // empty catch block
                    }
                    continue;
                }
            }
        }
    }

    public static void revertToSnapshot(VirtualMachine virtualMachine, String snapshotName) {
        ComputerUtils.revertToSnapshot(virtualMachine, snapshotName, null);
    }

    public static void revertToSnapshot(VirtualMachine virtualMachine, String snapshotName, @CheckForNull TaskListener listener) {
        IDomain domain;
        if (snapshotName != null && !snapshotName.isEmpty() && (domain = ComputerUtils.getDomain(virtualMachine, listener)) != null) {
            try {
                IDomainSnapshot snapshot = domain.snapshotLookupByName(snapshotName);
                try {
                    ComputerUtils.log(listener, MessageFormat.format("Reverting VM \"{0}\" to snapshot \"{1}\"", virtualMachine.getName(), snapshotName));
                    domain.revertToSnapshot(snapshot);
                }
                catch (VirtException e) {
                    ComputerUtils.error(listener, MessageFormat.format("Error reverting to snapshot \"{0}\" for VM \"{1}\": {2}", snapshotName, virtualMachine.getName(), e));
                }
            }
            catch (VirtException e) {
                ComputerUtils.error(listener, MessageFormat.format("No snapshot named \"{0}\" for VM \"{1}\": {2}", snapshotName, virtualMachine.getName(), e));
            }
        }
    }

    @CheckForNull
    private static IDomain getDomain(VirtualMachine virtualMachine, @CheckForNull TaskListener listener) {
        IDomain domain = null;
        try {
            Map<String, IDomain> domains = virtualMachine.getHypervisor().getDomains();
            domain = domains.get(virtualMachine.getName());
            if (domain == null) {
                ComputerUtils.error(listener, "No VM named \"" + virtualMachine.getName() + "\"");
            }
        }
        catch (VirtException e) {
            ComputerUtils.error(listener, "Can't get VM domains: " + String.valueOf(e));
        }
        return domain;
    }

    private static void log(@CheckForNull TaskListener listener, String message) {
        if (listener != null) {
            listener.getLogger().println(message);
        }
        LOGGER.log(Level.INFO, message);
    }

    private static void error(@CheckForNull TaskListener listener, String message) {
        if (listener != null) {
            listener.fatalError(message);
        }
        LOGGER.log(Level.WARNING, message);
    }
}

