/*
 * Decompiled with CFR 0.152.
 */
package com.github.terma.jenkins.githubprcoveragestatus;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.terma.jenkins.githubprcoveragestatus.GitUtils;
import com.github.terma.jenkins.githubprcoveragestatus.JsonUtils;
import com.github.terma.jenkins.githubprcoveragestatus.MasterCoverageRepository;
import java.io.IOException;
import java.io.PrintStream;
import java.net.URLEncoder;
import java.text.MessageFormat;
import java.util.List;
import org.apache.commons.httpclient.Credentials;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScheme;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.auth.BasicScheme;
import org.apache.commons.httpclient.methods.GetMethod;

public class SonarMasterCoverageRepository
implements MasterCoverageRepository {
    private static final String SONAR_SEARCH_PROJECTS_API_PATH = "/api/projects/index";
    private static final String SONAR_COMPONENT_MEASURE_API_PATH = "/api/measures/component";
    public static final String SONAR_OVERALL_LINE_COVERAGE_METRIC_NAME = "coverage";
    private final String sonarUrl;
    private final String login;
    private final HttpClient httpClient;
    private final ObjectMapper objectMapper = new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
    private PrintStream buildLog;

    public SonarMasterCoverageRepository(String sonarUrl, String login, String password, PrintStream buildLog) {
        this.sonarUrl = sonarUrl;
        this.login = login;
        this.buildLog = buildLog;
        this.httpClient = new HttpClient();
        if (login != null) {
            this.httpClient.getState().setCredentials(AuthScope.ANY, (Credentials)new UsernamePasswordCredentials(login, password));
        }
    }

    @Override
    public float get(String gitHubRepoUrl) {
        String repoName = GitUtils.getRepoName(gitHubRepoUrl);
        this.log("Getting coverage for Git Repo URL: %s by repo name: %s", gitHubRepoUrl, repoName);
        try {
            SonarProject sonarProject = this.getSonarProject(repoName);
            return this.getCoverageMeasure(sonarProject);
        }
        catch (Exception e) {
            this.log("Failed to get master coverage for %s", gitHubRepoUrl);
            this.log("Exception message '%s'", e);
            e.printStackTrace(this.buildLog);
            return 0.0f;
        }
    }

    private SonarProject getSonarProject(String repoName) throws SonarProjectRetrievalException {
        try {
            String searchUri = this.sonarUrl + SONAR_SEARCH_PROJECTS_API_PATH + "?search=" + repoName;
            GetMethod method = this.executeGetRequest(searchUri);
            List sonarProjects = (List)this.objectMapper.readValue(method.getResponseBodyAsStream(), (TypeReference)new TypeReference<List<SonarProject>>(){});
            if (sonarProjects.isEmpty()) {
                throw new SonarProjectRetrievalException("No sonar project found for repo" + repoName);
            }
            if (sonarProjects.size() == 1) {
                this.log("Found project for repo name %s - %s", repoName, sonarProjects.get(0));
                return (SonarProject)sonarProjects.get(0);
            }
            this.log("Found multiple projects for repo name %s - found %s - returning first result", repoName, sonarProjects);
            return (SonarProject)sonarProjects.get(0);
        }
        catch (Exception e) {
            throw new SonarProjectRetrievalException(String.format("failed to search for sonar project %s - %s", repoName, e.getMessage()), e);
        }
    }

    private float getCoverageMeasure(SonarProject project) throws SonarCoverageMeasureRetrievalException {
        String uri = MessageFormat.format("{0}{1}?componentKey={2}&metricKeys={3}", this.sonarUrl, SONAR_COMPONENT_MEASURE_API_PATH, URLEncoder.encode(project.getKey()), SONAR_OVERALL_LINE_COVERAGE_METRIC_NAME);
        try {
            GetMethod method = this.executeGetRequest(uri);
            String value = (String)JsonUtils.findInJson(method.getResponseBodyAsString(), "component.measures[0].value");
            return Float.parseFloat(value) / 100.0f;
        }
        catch (Exception e) {
            throw new SonarCoverageMeasureRetrievalException(String.format("failed to get coverage measure for sonar project %s - %s", project.getKey(), e.getMessage()), e);
        }
    }

    private GetMethod executeGetRequest(String uri) throws IOException, HttpClientException {
        int status;
        GetMethod method = new GetMethod(uri);
        if (this.login != null) {
            method.getHostAuthState().setAuthScheme((AuthScheme)new BasicScheme());
        }
        if ((status = this.httpClient.executeMethod((HttpMethod)method)) >= 400) {
            throw new HttpClientException(uri, status, method.getResponseBodyAsString());
        }
        return method;
    }

    private void log(String format, Object ... arguments) {
        this.buildLog.printf(format, arguments);
        this.buildLog.println();
    }

    private static class SonarCoverageMeasureRetrievalException
    extends Exception {
        private SonarCoverageMeasureRetrievalException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    private static class SonarProjectRetrievalException
    extends Exception {
        private SonarProjectRetrievalException(String message) {
            super(message);
        }

        private SonarProjectRetrievalException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    private static class SonarProject {
        @JsonProperty(value="k")
        String key;

        private SonarProject() {
        }

        String getKey() {
            return this.key;
        }

        void setKey(String key) {
            this.key = key;
        }

        public String toString() {
            return MessageFormat.format("({0}(key = {1}))", this.getClass().getSimpleName(), this.key);
        }
    }

    private static class HttpClientException
    extends Exception {
        HttpClientException(String uri, int status, String reason) {
            super("request to " + uri + " failed with " + status + " reason " + reason);
        }
    }
}

