/**
 * @author Dominik Bartholdi (imod)
 * @author Thomas Doering (thomas-dee) Licensed under the MIT License. See License.txt in the
 *     project root for license information.
 */
package org.jenkinsci.plugins.badge.actions;

import hudson.Extension;
import hudson.ExtensionList;
import hudson.model.Item;
import hudson.model.Job;
import hudson.model.Run;
import hudson.model.UnprotectedRootAction;
import hudson.security.ACL;
import hudson.security.ACLContext;
import hudson.security.Permission;
import hudson.security.PermissionScope;
import hudson.util.HttpResponses;
import java.io.IOException;
import jenkins.model.Jenkins;
import org.jenkinsci.plugins.badge.*;
import org.jenkinsci.plugins.badge.extensionpoints.InternalRunSelectorExtensionPoint;
import org.jenkinsci.plugins.badge.extensionpoints.JobSelectorExtensionPoint;
import org.jenkinsci.plugins.badge.extensionpoints.RunSelectorExtensionPoint;
import org.kohsuke.stapler.HttpResponse;
import org.kohsuke.stapler.QueryParameter;
import org.kohsuke.stapler.StaplerRequest2;
import org.kohsuke.stapler.StaplerResponse2;
import org.kohsuke.stapler.WebMethod;

/**
 * Exposes the build status badge via unprotected URL.
 *
 * <p>Even though the URL is unprotected, the user does still need the 'ViewStatus' permission on
 * the given Job. If you want the status icons to be public readable/accessible, just grant the
 * 'ViewStatus' permission globally to 'anonymous'.
 */
@SuppressWarnings("rawtypes")
@Extension
public class PublicBuildStatusAction implements UnprotectedRootAction {
    public static final Permission VIEW_STATUS = new Permission(
            Item.PERMISSIONS, "ViewStatus", Messages._ViewStatus_Permission(), Item.READ, PermissionScope.ITEM);
    private static final Jenkins jInstance = Jenkins.get();
    private IconRequestHandler iconRequestHandler;

    public PublicBuildStatusAction() throws IOException {
        iconRequestHandler = new IconRequestHandler();
    }

    @Override
    public String getUrlName() {
        return "buildStatus";
    }

    @Override
    public String getIconFileName() {
        return null;
    }

    @Override
    public String getDisplayName() {
        return null;
    }

    @WebMethod(name = "icon")
    public HttpResponse doIcon(
            StaplerRequest2 req,
            StaplerResponse2 rsp,
            @QueryParameter String job,
            @QueryParameter String build,
            @QueryParameter String style,
            @QueryParameter String subject,
            @QueryParameter String status,
            @QueryParameter String color,
            @QueryParameter String animatedOverlayColor,
            @QueryParameter String config,
            @QueryParameter String link) {
        if (job == null) {
            return PluginImpl.iconRequestHandler.handleIconRequest(
                    style, subject, status, color, animatedOverlayColor, link);
        } else {
            Job<?, ?> project = getProject(job, false);
            if (build != null && project != null) {
                Run<?, ?> run = getRun(project, build, false);
                return iconRequestHandler.handleIconRequestForRun(
                        run, style, subject, status, color, animatedOverlayColor, config, link);
            } else {
                return iconRequestHandler.handleIconRequestForJob(
                        project, style, subject, status, color, animatedOverlayColor, config, link);
            }
        }
    }

    @WebMethod(name = "icon.svg")
    public HttpResponse doIconDotSvg(
            StaplerRequest2 req,
            StaplerResponse2 rsp,
            @QueryParameter String job,
            @QueryParameter String build,
            @QueryParameter String style,
            @QueryParameter String subject,
            @QueryParameter String status,
            @QueryParameter String color,
            @QueryParameter String animatedOverlayColor,
            @QueryParameter String config,
            @QueryParameter String link) {
        return doIcon(req, rsp, job, build, style, subject, status, color, animatedOverlayColor, config, link);
    }

    public String doText(
            StaplerRequest2 req, StaplerResponse2 rsp, @QueryParameter String job, @QueryParameter String build) {
        if (job == null) {
            return "Missing query parameter: job";
        }

        Job<?, ?> project = getProject(job, true);
        if (build != null) {
            Run<?, ?> run = getRun(project, build, true);
            return run.getIconColor().getDescription();
        } else {
            return project.getIconColor().getDescription();
        }
    }

    private static Job<?, ?> getProject(String job, Boolean throwErrorWhenNotFound) {
        Job<?, ?> p = null;
        if (job != null) {
            // as the user might have ViewStatus permission only (e.g. as anonymous) we get the
            // project impersonate and check for permission after getting the project

            try (ACLContext ctx = ACL.as2(ACL.SYSTEM2)) {
                // first try to get Job via JobSelectorExtensionPoints
                for (JobSelectorExtensionPoint jobSelector : ExtensionList.lookup(JobSelectorExtensionPoint.class)) {
                    p = jobSelector.select(job);
                    if (p != null) {
                        break;
                    }
                }

                if (p == null) {
                    p = jInstance.getItemByFullName(job, Job.class);
                }
            }
        }

        // check if user has permission to view the status
        if (p == null || !p.hasPermission(VIEW_STATUS)) {
            if (throwErrorWhenNotFound) {
                throw HttpResponses.notFound();
            }
            return null;
        }

        return p;
    }

    public static Run<?, ?> getRun(Job<?, ?> project, String build, Boolean throwErrorWhenNotFound) {
        Run<?, ?> run = null;

        if (project != null && build != null) {
            // as the user might have ViewStatus permission only (e.g. as anonymous) we get the
            // project impersonate and check for permission after getting the project
            try (ACLContext ctx = ACL.as2(ACL.SYSTEM2)) {
                for (String token : build.split(",")) {
                    Run newRun = null;
                    // first: try to get Run via our InternalRunSelectorExtensionPoints
                    for (InternalRunSelectorExtensionPoint runSelector :
                            ExtensionList.lookup(InternalRunSelectorExtensionPoint.class)) {
                        newRun = runSelector.select(project, token, run);
                        if (newRun != null) {
                            break;
                        }
                    }

                    if (newRun == null) {
                        // second: try to get Run via RunSelectorExtensionPoints
                        for (RunSelectorExtensionPoint runSelector :
                                ExtensionList.lookup(RunSelectorExtensionPoint.class)) {
                            newRun = runSelector.select(project, token, run);
                            if (newRun != null) {
                                break;
                            }
                        }
                    }

                    if (newRun != null) {
                        run = newRun;
                    } else {
                        break;
                    }
                }
            }
        }
        // check if user has permission to view the status
        if (run == null || !run.hasPermission(VIEW_STATUS)) {
            if (throwErrorWhenNotFound) {
                throw HttpResponses.notFound();
            }
            return null;
        }

        return run;
    }
}
