/*
 * Decompiled with CFR 0.152.
 */
package org.jenkinsci.plugins.docker.workflow.client;

import com.google.common.base.Optional;
import edu.umd.cs.findbugs.annotations.CheckForNull;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import hudson.EnvVars;
import hudson.FilePath;
import hudson.Launcher;
import hudson.model.Node;
import hudson.model.TaskListener;
import hudson.util.ArgumentListBuilder;
import hudson.util.VersionNumber;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.StringReader;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang.StringUtils;
import org.jenkinsci.plugins.docker.commons.fingerprint.ContainerRecord;
import org.jenkinsci.plugins.docker.commons.tools.DockerTool;
import org.jenkinsci.plugins.docker.workflow.client.ControlGroup;
import org.jenkinsci.plugins.docker.workflow.client.LaunchResult;
import org.kohsuke.accmod.Restricted;
import org.kohsuke.accmod.restrictions.NoExternalUse;

public class DockerClient {
    private static final Logger LOGGER = Logger.getLogger(DockerClient.class.getName());
    @Restricted(value={NoExternalUse.class})
    @SuppressFBWarnings(value={"MS_SHOULD_BE_FINAL"}, justification="mutable for scripts")
    public static int CLIENT_TIMEOUT = Integer.getInteger(DockerClient.class.getName() + ".CLIENT_TIMEOUT", 180);
    @Restricted(value={NoExternalUse.class})
    @SuppressFBWarnings(value={"MS_SHOULD_BE_FINAL"}, justification="mutable for scripts")
    public static boolean SKIP_RM_ON_STOP = Boolean.getBoolean(DockerClient.class.getName() + ".SKIP_RM_ON_STOP");
    public static final String DOCKER_DATE_TIME_FORMAT = "yyyy-MM-dd'T'HH:mm:ss";
    private final Launcher launcher;
    @CheckForNull
    private final Node node;
    @CheckForNull
    private final String toolName;
    private static final Pattern pattern = Pattern.compile("^(\\D+)(\\d+)\\.(\\d+)\\.(\\d+)(.*)");
    private static final Pattern hostnameMount = Pattern.compile("/containers/([a-z0-9]{64})/hostname");

    public DockerClient(@NonNull Launcher launcher, @CheckForNull Node node, @CheckForNull String toolName) {
        this.launcher = launcher;
        this.node = node;
        this.toolName = toolName;
    }

    public String run(@NonNull EnvVars launchEnv, @NonNull String image, @CheckForNull String args, @CheckForNull String workdir, @NonNull Map<String, String> volumes, @NonNull Collection<String> volumesFromContainers, @NonNull EnvVars containerEnv, @NonNull String user, String ... command) throws IOException, InterruptedException {
        ArgumentListBuilder argb = new ArgumentListBuilder();
        argb.add(new String[]{"run", "-t", "-d"});
        if (StringUtils.isNotEmpty((String)user)) {
            argb.add(new String[]{"-u", user});
        }
        if (args != null) {
            argb.addTokenized(args);
        }
        if (workdir != null) {
            argb.add(new String[]{"-w", workdir});
        }
        for (Map.Entry<String, String> volume : volumes.entrySet()) {
            argb.add(new String[]{"-v", volume.getKey() + ":" + volume.getValue() + ":rw,z"});
        }
        for (String containerId : volumesFromContainers) {
            argb.add(new String[]{"--volumes-from", containerId});
        }
        for (Map.Entry variable : containerEnv.entrySet()) {
            argb.add("-e");
            argb.addMasked((String)variable.getKey() + "=" + (String)variable.getValue());
        }
        argb.add(image).add(command);
        LaunchResult result = this.launch(launchEnv, false, null, argb);
        if (result.getStatus() == 0) {
            return result.getOut();
        }
        throw new IOException(String.format("Failed to run image '%s'. Error: %s", image, result.getErr()));
    }

    public List<String> listProcess(@NonNull EnvVars launchEnv, @NonNull String containerId) throws IOException, InterruptedException {
        LaunchResult result = this.launch(launchEnv, false, "top", containerId, "-eo", "pid,comm");
        if (result.getStatus() != 0) {
            throw new IOException(String.format("Failed to run top '%s'. Error: %s", containerId, result.getErr()));
        }
        ArrayList<String> processes = new ArrayList<String>();
        try (StringReader r = new StringReader(result.getOut());
             BufferedReader in = new BufferedReader(r);){
            String line;
            in.readLine();
            while ((line = in.readLine()) != null) {
                StringTokenizer stringTokenizer = new StringTokenizer(line, " ");
                if (stringTokenizer.countTokens() < 2) {
                    throw new IOException("Unexpected `docker top` output : " + line);
                }
                stringTokenizer.nextToken();
                processes.add(stringTokenizer.nextToken());
            }
        }
        return processes;
    }

    public void stop(@NonNull EnvVars launchEnv, @NonNull String containerId) throws IOException, InterruptedException {
        LaunchResult result = this.launch(launchEnv, false, "stop", "--time=1", containerId);
        if (result.getStatus() != 0) {
            throw new IOException(String.format("Failed to kill container '%s'.", containerId));
        }
        if (!SKIP_RM_ON_STOP) {
            this.rm(launchEnv, containerId);
        }
    }

    public void rm(@NonNull EnvVars launchEnv, @NonNull String containerId) throws IOException, InterruptedException {
        LaunchResult result = this.launch(launchEnv, false, "rm", "-f", "--volumes", containerId);
        if (result.getStatus() != 0) {
            throw new IOException(String.format("Failed to rm container '%s'.", containerId));
        }
    }

    @CheckForNull
    public String inspect(@NonNull EnvVars launchEnv, @NonNull String objectId, @NonNull String fieldPath) throws IOException, InterruptedException {
        LaunchResult result = this.launch(launchEnv, true, "inspect", "-f", String.format("{{%s}}", fieldPath), objectId);
        if (result.getStatus() == 0) {
            return result.getOut();
        }
        return null;
    }

    @NonNull
    public String inspectRequiredField(@NonNull EnvVars launchEnv, @NonNull String objectId, @NonNull String fieldPath) throws IOException, InterruptedException {
        String fieldValue = this.inspect(launchEnv, objectId, fieldPath);
        if (fieldValue == null) {
            throw new IOException("Cannot retrieve " + fieldPath + " from 'docker inspect " + objectId + "'");
        }
        return fieldValue;
    }

    @CheckForNull
    private Date getCreatedDate(@NonNull EnvVars launchEnv, @NonNull String objectId) throws IOException, InterruptedException {
        String createdString = this.inspect(launchEnv, objectId, "json .Created");
        if (createdString == null) {
            return null;
        }
        String s = createdString.substring(1, DOCKER_DATE_TIME_FORMAT.length() - 1);
        try {
            return new SimpleDateFormat(DOCKER_DATE_TIME_FORMAT).parse(s);
        }
        catch (ParseException e) {
            throw new IOException(String.format("Error parsing created date '%s' for object '%s'.", s, objectId), e);
        }
    }

    @CheckForNull
    public VersionNumber version() throws IOException, InterruptedException {
        LaunchResult result = this.launch(new EnvVars(), true, "-v");
        if (result.getStatus() == 0) {
            return DockerClient.parseVersionNumber(result.getOut());
        }
        return null;
    }

    protected static VersionNumber parseVersionNumber(@NonNull String versionString) {
        Matcher matcher = pattern.matcher(versionString.trim());
        if (matcher.matches()) {
            String major = matcher.group(2);
            String minor = matcher.group(3);
            String maint = matcher.group(4);
            return new VersionNumber(String.format("%s.%s.%s", major, minor, maint));
        }
        return null;
    }

    private LaunchResult launch(@NonNull EnvVars launchEnv, boolean quiet, String ... args) throws IOException, InterruptedException {
        return this.launch(launchEnv, quiet, (FilePath)null, args);
    }

    private LaunchResult launch(@NonNull EnvVars launchEnv, boolean quiet, FilePath pwd, String ... args) throws IOException, InterruptedException {
        return this.launch(launchEnv, quiet, pwd, new ArgumentListBuilder(args));
    }

    private LaunchResult launch(@NonNull EnvVars launchEnv, boolean quiet, FilePath pwd, @NonNull ArgumentListBuilder args) throws IOException, InterruptedException {
        args.prepend(new String[]{DockerTool.getExecutable((String)this.toolName, (Node)this.node, (TaskListener)this.launcher.getListener(), (EnvVars)launchEnv)});
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE, "Executing docker command {0}", args.toString());
        }
        Launcher.ProcStarter procStarter = this.launcher.launch();
        if (pwd != null) {
            procStarter.pwd(pwd);
        }
        LaunchResult result = new LaunchResult();
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        ByteArrayOutputStream err = new ByteArrayOutputStream();
        result.setStatus(procStarter.quiet(quiet).cmds(args).envs((Map)launchEnv).stdout((OutputStream)out).stderr((OutputStream)err).start().joinWithTimeout((long)CLIENT_TIMEOUT, TimeUnit.SECONDS, this.launcher.getListener()));
        String charsetName = Charset.defaultCharset().name();
        result.setOut(out.toString(charsetName));
        result.setErr(err.toString(charsetName));
        return result;
    }

    public String whoAmI() throws IOException, InterruptedException {
        if (!this.launcher.isUnix()) {
            return "";
        }
        ByteArrayOutputStream userId = new ByteArrayOutputStream();
        this.launcher.launch().cmds(new String[]{"id", "-u"}).quiet(true).stdout((OutputStream)userId).start().joinWithTimeout((long)CLIENT_TIMEOUT, TimeUnit.SECONDS, this.launcher.getListener());
        ByteArrayOutputStream groupId = new ByteArrayOutputStream();
        this.launcher.launch().cmds(new String[]{"id", "-g"}).quiet(true).stdout((OutputStream)groupId).start().joinWithTimeout((long)CLIENT_TIMEOUT, TimeUnit.SECONDS, this.launcher.getListener());
        String charsetName = Charset.defaultCharset().name();
        return String.format("%s:%s", userId.toString(charsetName).trim(), groupId.toString(charsetName).trim());
    }

    public Optional<String> getContainerIdIfContainerized() throws IOException, InterruptedException {
        Optional<String> containerId;
        if (this.node == null) {
            return Optional.absent();
        }
        FilePath cgroupFile = this.node.createPath("/proc/self/cgroup");
        if (cgroupFile != null && cgroupFile.exists() && (containerId = ControlGroup.getContainerId(cgroupFile)).isPresent()) {
            return containerId;
        }
        FilePath mountInfo = this.node.createPath("/proc/1/mountinfo");
        if (mountInfo != null && mountInfo.exists()) {
            try (InputStream is = mountInfo.read();
                 InputStreamReader r = new InputStreamReader(is, StandardCharsets.UTF_8);
                 BufferedReader br = new BufferedReader(r);){
                while (true) {
                    String line;
                    if ((line = br.readLine()) != null) {
                        Matcher m = hostnameMount.matcher(line);
                        if (!m.find()) continue;
                        Optional optional = Optional.of((Object)m.group(1));
                        return optional;
                        continue;
                    }
                    break;
                }
            }
        }
        return Optional.absent();
    }

    public ContainerRecord getContainerRecord(@NonNull EnvVars launchEnv, String containerId) throws IOException, InterruptedException {
        String host = this.inspectRequiredField(launchEnv, containerId, ".Config.Hostname");
        String containerName = this.inspectRequiredField(launchEnv, containerId, ".Name");
        Date created = this.getCreatedDate(launchEnv, containerId);
        String image = this.inspectRequiredField(launchEnv, containerId, ".Image");
        return new ContainerRecord(host, containerId, image, containerName, created != null ? created.getTime() : 0L, Collections.emptyMap());
    }

    public List<String> getVolumes(@NonNull EnvVars launchEnv, String containerID) throws IOException, InterruptedException {
        LaunchResult result = this.launch(launchEnv, true, "inspect", "-f", "{{range.Mounts}}{{.Destination}}\n{{end}}", containerID);
        if (result.getStatus() != 0) {
            return Collections.emptyList();
        }
        String volumes = result.getOut();
        if (volumes.isEmpty()) {
            return Collections.emptyList();
        }
        return Arrays.asList(volumes.replace("\\", "/").split("\\n"));
    }
}

