/*
 * This file is part of Dependency-Check Jenkins plugin.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jenkinsci.plugins.DependencyCheck.model;

import java.io.Serial;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Java Bean class for Vulnerability found by DependencyCheck.
 *
 * @author Steve Springett (steve.springett@owasp.org)
 * @since 5.0.0
 */
public class Vulnerability implements Serializable {

    @Serial
    private static final long serialVersionUID = 2146048369342617342L;

    public enum Source {
        NVD,
        NPM,
        RETIREJS,
        OSSINDEX,
        BUNDLEAUDIT
    }

    private Source source = null;
    private String name;
    private String description;
    private String severity;
    private CvssV2 cvssV2;
    private CvssV3 cvssV3;
    private List<String> cwes = new ArrayList<>();
    private List<Reference> references = new ArrayList<>();

    public Source getSource() {
        return source;
    }

    public void setSource(Source source) {
        this.source = source;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public String getSeverity() {
        return severity;
    }

    public void setSeverity(String severity) {
        this.severity = severity;
    }

    public CvssV2 getCvssV2() {
        return cvssV2;
    }

    public void setCvssV2(CvssV2 cvssV2) {
        this.cvssV2 = cvssV2;
    }

    public CvssV3 getCvssV3() {
        return cvssV3;
    }

    public void setCvssV3(CvssV3 cvssV3) {
        this.cvssV3 = cvssV3;
    }

    public List<String> getCwes() {
        return cwes;
    }

    public void setCwes(List<String> cwes) {
        this.cwes = cwes;
    }

    public void addCwe(String cwe) {
        cwes.add(cwe);
    }

    public List<Reference> getReferences() {
        return references;
    }

    public void setReferences(List<Reference> references) {
        this.references = references;
    }

    public void addReference(Reference reference) {
        references.add(reference);
    }

    @Override
    public int hashCode() {
        return Objects.hash(name, severity, source);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        Vulnerability other = (Vulnerability) obj;
        return Objects.equals(name, other.name) && Objects.equals(severity, other.severity) && source == other.source;
    }

    @Override
    public String toString() {
        return name + " " + source;
    }
}
