/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.jenkins.containeragents.aci;

import com.azure.resourcemanager.AzureResourceManager;
import com.azure.resourcemanager.resources.models.Deployment;
import com.azure.resourcemanager.resources.models.GenericResource;
import com.microsoft.jenkins.containeragents.aci.AciCloud;
import com.microsoft.jenkins.containeragents.aci.AciComputer;
import com.microsoft.jenkins.containeragents.aci.AciService;
import com.microsoft.jenkins.containeragents.util.AzureContainerUtils;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import hudson.Extension;
import hudson.model.AsyncPeriodicWork;
import hudson.model.Computer;
import hudson.model.TaskListener;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.nio.file.Paths;
import java.time.OffsetDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.logging.Level;
import java.util.logging.Logger;
import jenkins.model.Jenkins;
import org.apache.commons.lang3.StringUtils;

@Extension
public class AciCleanTask
extends AsyncPeriodicWork {
    private static final Logger LOGGER = Logger.getLogger(AciCleanTask.class.getName());
    private static final int RECURRENCE_PERIOD_IN_MILLIS = 1800000;
    private static final long SUCCESSFUL_DEPLOYMENT_TIMEOUT_IN_MINUTES = 60L;
    private static final long FAILING_DEPLOYMENT_TIMEOUT_IN_MINUTES = 480L;

    public AciCleanTask() {
        super("ACI Period Clean Task");
    }

    private void clean() {
        Jenkins instance = Jenkins.getInstanceOrNull();
        if (instance == null) {
            return;
        }
        for (AciCloud cloud : instance.clouds.getAll(AciCloud.class)) {
            this.cleanLeakedContainer(cloud);
        }
    }

    public static String loadProperty(String name) {
        String value = System.getProperty(name);
        if (StringUtils.isBlank((CharSequence)value)) {
            return AciCleanTask.loadEnv(name);
        }
        return value;
    }

    public static String loadEnv(String name) {
        String value = System.getenv(name);
        if (StringUtils.isBlank((CharSequence)value)) {
            return "";
        }
        return value;
    }

    public AciCloud getCloud(String cloudName) {
        return Jenkins.getInstanceOrNull() == null ? null : (AciCloud)Jenkins.get().getCloud(cloudName);
    }

    public void cleanDeployments() {
        this.cleanDeployments(60L, 480L);
    }

    @SuppressFBWarnings(value={"DCN_NULLPOINTER_EXCEPTION"}, justification="TODO needs triage")
    private void cleanDeployments(long successTimeoutInMinutes, long failTimeoutInMinutes) {
        DeploymentInfo firstBackInQueue = null;
        ConcurrentLinkedQueue<DeploymentInfo> deploymentsToClean = DeploymentRegistrar.getInstance().getDeploymentsToClean();
        while (!deploymentsToClean.isEmpty() && firstBackInQueue != deploymentsToClean.peek()) {
            DeploymentInfo info = (DeploymentInfo)deploymentsToClean.remove();
            LOGGER.log(this.getNormalLoggingLevel(), "AzureAciCleanUpTask: cleanDeployments: Checking deployment {0}", info.getDeploymentName());
            AciCloud cloud = this.getCloud(info.getCloudName());
            if (cloud == null) continue;
            try {
                Deployment deployment;
                AzureResourceManager azureClient = AzureContainerUtils.getAzureClient(cloud.getCredentialsId());
                try {
                    deployment = (Deployment)azureClient.deployments().getByResourceGroup(info.getResourceGroupName(), info.getDeploymentName());
                }
                catch (NullPointerException e) {
                    LOGGER.log(this.getNormalLoggingLevel(), "AzureAciCleanUpTask: cleanDeployments: Deployment not found, skipping");
                    continue;
                }
                if (deployment == null) {
                    LOGGER.log(this.getNormalLoggingLevel(), "AzureAciCleanUpTask: cleanDeployments: Deployment not found, skipping");
                    continue;
                }
                OffsetDateTime deploymentTime = deployment.timestamp();
                LOGGER.log(this.getNormalLoggingLevel(), "AzureAciCleanUpTask: cleanDeployments: Deployment created on {0}", deploymentTime.toString());
                long diffTimeInMinutes = ChronoUnit.MINUTES.between(deploymentTime, OffsetDateTime.now());
                String state = deployment.provisioningState();
                if (!"succeeded".equalsIgnoreCase(state) && diffTimeInMinutes > failTimeoutInMinutes) {
                    LOGGER.log(this.getNormalLoggingLevel(), "AzureAciCleanUpTask: cleanDeployments: Failed deployment older than {0} minutes, deleting", failTimeoutInMinutes);
                    azureClient.deployments().deleteByResourceGroup(info.getResourceGroupName(), info.getDeploymentName());
                    continue;
                }
                if ("succeeded".equalsIgnoreCase(state) && diffTimeInMinutes > successTimeoutInMinutes) {
                    LOGGER.log(this.getNormalLoggingLevel(), "AzureAciCleanUpTask: cleanDeployments: Successful deployment older than {0} minutes, deleting", successTimeoutInMinutes);
                    azureClient.deployments().deleteByResourceGroup(info.getResourceGroupName(), info.getDeploymentName());
                    continue;
                }
                LOGGER.log(this.getNormalLoggingLevel(), "AzureAciCleanUpTask: cleanDeployments: Deployment newer than timeout, keeping");
                if (firstBackInQueue == null) {
                    firstBackInQueue = info;
                }
                deploymentsToClean.add(info);
            }
            catch (Exception e) {
                LOGGER.log(Level.WARNING, "AzureAciCleanUpTask: cleanDeployments: Failed to get/delete deployment", e);
                if (!info.hasAttemptsRemaining()) continue;
                info.decrementAttemptsRemaining();
                if (firstBackInQueue == null) {
                    firstBackInQueue = info;
                }
                deploymentsToClean.add(info);
            }
        }
        DeploymentRegistrar.getInstance().syncDeploymentsToClean();
    }

    private void cleanLeakedContainer(AciCloud cloud) {
        AzureResourceManager azureClient;
        LOGGER.log(Level.INFO, "Starting to clean leaked containers for cloud " + cloud.getName());
        try {
            azureClient = cloud.getAzureClient();
        }
        catch (Exception e) {
            return;
        }
        String resourceGroup = cloud.getResourceGroup();
        String credentialsId = cloud.getCredentialsId();
        if (StringUtils.isBlank((CharSequence)resourceGroup) || StringUtils.isBlank((CharSequence)credentialsId)) {
            return;
        }
        Set<String> validContainerSet = this.getValidContainer();
        for (GenericResource resource : azureClient.genericResources().listByResourceGroup(resourceGroup)) {
            if (!resource.resourceProviderNamespace().equalsIgnoreCase("Microsoft.ContainerInstance") || !resource.resourceType().equalsIgnoreCase("containerGroups") || !resource.tags().containsKey("JenkinsInstance") || !((String)resource.tags().get("JenkinsInstance")).equalsIgnoreCase(Jenkins.get().getLegacyInstanceId()) || validContainerSet.contains(resource.name())) continue;
            AciCloud.getThreadPool().submit(() -> AciService.deleteAciContainerGroup(credentialsId, resourceGroup, resource.name(), null));
        }
    }

    private Set<String> getValidContainer() {
        TreeSet<String> result = new TreeSet<String>();
        for (Computer computer : Jenkins.get().getComputers()) {
            if (!(computer instanceof AciComputer)) continue;
            result.add(computer.getName());
        }
        return result;
    }

    public void execute(TaskListener arg0) {
        this.clean();
        this.cleanDeployments();
    }

    public long getRecurrencePeriod() {
        return 1800000L;
    }

    public static class DeploymentRegistrar {
        private static final String OUTPUT_FILE = Paths.get(AciCleanTask.loadProperty("JENKINS_HOME"), "aci-deployment.out").toString();
        private static final DeploymentRegistrar DEPLOYMENT_REGISTRAR = new DeploymentRegistrar();
        private static final int MAX_DELETE_ATTEMPTS = 3;
        private ConcurrentLinkedQueue<DeploymentInfo> deploymentsToClean = new ConcurrentLinkedQueue();

        protected DeploymentRegistrar() {
            try (ObjectInputStream ois = new ObjectInputStream(new FileInputStream(OUTPUT_FILE));){
                this.deploymentsToClean = (ConcurrentLinkedQueue)ois.readObject();
            }
            catch (FileNotFoundException e) {
                LOGGER.log(Level.WARNING, "AzureAciCleanUpTask: readResolve: Cannot open deployment output file");
            }
            catch (Exception e) {
                LOGGER.log(Level.WARNING, "AzureAciCleanUpTask: readResolve: Cannot deserialize deploymentsToClean", e);
            }
        }

        public static DeploymentRegistrar getInstance() {
            return DEPLOYMENT_REGISTRAR;
        }

        public ConcurrentLinkedQueue<DeploymentInfo> getDeploymentsToClean() {
            return this.deploymentsToClean;
        }

        public void registerDeployment(String cloudName, String resourceGroupName, String deploymentName) {
            LOGGER.log(Level.INFO, "AzureAciCleanUpTask: registerDeployment: Registering deployment {0} in {1}", new Object[]{deploymentName, resourceGroupName});
            DeploymentInfo newDeploymentToClean = new DeploymentInfo(cloudName, resourceGroupName, deploymentName, 3);
            this.deploymentsToClean.add(newDeploymentToClean);
            this.syncDeploymentsToClean();
        }

        public synchronized void syncDeploymentsToClean() {
            try (ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(OUTPUT_FILE));){
                oos.writeObject(this.deploymentsToClean);
            }
            catch (FileNotFoundException e) {
                LOGGER.log(Level.WARNING, "AzureAciCleanUpTask: registerDeployment: Cannot open deployment output file" + OUTPUT_FILE);
            }
            catch (IOException e) {
                LOGGER.log(Level.WARNING, "AzureAciCleanUpTask: registerDeployment: Serialize failed", e);
            }
        }
    }

    private static class DeploymentInfo
    implements Serializable {
        private final String cloudName;
        private final String deploymentName;
        private final String resourceGroupName;
        private int attemptsRemaining;

        DeploymentInfo(String cloudName, String resourceGroupName, String deploymentName, int deleteAttempts) {
            this.cloudName = cloudName;
            this.deploymentName = deploymentName;
            this.resourceGroupName = resourceGroupName;
            this.attemptsRemaining = deleteAttempts;
        }

        String getCloudName() {
            return this.cloudName;
        }

        String getDeploymentName() {
            return this.deploymentName;
        }

        String getResourceGroupName() {
            return this.resourceGroupName;
        }

        boolean hasAttemptsRemaining() {
            return this.attemptsRemaining > 0;
        }

        void decrementAttemptsRemaining() {
            --this.attemptsRemaining;
        }
    }
}

