/*
 * Decompiled with CFR 0.152.
 */
package io.jenkins.plugins;

import hudson.EnvVars;
import hudson.FilePath;
import hudson.Launcher;
import hudson.model.Run;
import hudson.model.TaskListener;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Map;

final class LocalRunExecutor {
    private static final String API_KEY_ENV = "SEDSTART_API_KEY";
    private static final String DIR_NAME = ".sedstart";
    private static final String BIN_NAME = "sedstart";

    LocalRunExecutor() {
    }

    void execute(Run<?, ?> run, FilePath workspace, EnvVars env, Launcher launcher, TaskListener listener, Integer projectId, Integer suiteId, Integer testId, Integer profileId, String browser, boolean headless, String environment) throws IOException, InterruptedException {
        LocalRunExecutor.log(listener, "Starting local execution");
        LocalRunExecutor.log(listener, "Validating inputs");
        LocalRunExecutor.validate(projectId, suiteId, testId, profileId);
        String apiKey = (String)env.get((Object)API_KEY_ENV);
        if (apiKey == null || apiKey.isBlank()) {
            throw new IOException("SEDSTART_API_KEY is not set. Use Jenkins credentials binding.");
        }
        LocalRunExecutor.log(listener, "Preparing .sedstart directory");
        FilePath sedstartDir = workspace.child(DIR_NAME);
        sedstartDir.mkdirs();
        String os = LocalRunExecutor.detectOs(launcher, listener);
        String arch = LocalRunExecutor.detectArch(launcher, listener);
        LocalRunExecutor.log(listener, "Platform resolved: " + os + " / " + arch);
        FilePath binary = sedstartDir.child(os.equals("windows") ? "sedstart.exe" : BIN_NAME);
        FilePath envFile = sedstartDir.child("default.env");
        if (!envFile.exists()) {
            LocalRunExecutor.log(listener, "Creating default.env");
            envFile.touch(0L);
        }
        if (!binary.exists()) {
            LocalRunExecutor.log(listener, "sedstart binary not found, downloading");
            LocalRunExecutor.downloadBinary(binary, launcher, environment, os, arch, listener);
            LocalRunExecutor.chmodIfNeeded(binary, launcher, os, listener);
        } else {
            LocalRunExecutor.log(listener, "sedstart binary already present, reusing");
        }
        LocalRunExecutor.log(listener, "Building sedstart command");
        ArrayList<String> cmd = new ArrayList<String>();
        boolean addHeadlessFlag = false;
        if (headless) {
            addHeadlessFlag = true;
            LocalRunExecutor.log(listener, "Headless enabled \u2192 running with -q");
        } else {
            LocalRunExecutor.log(listener, "Headless disabled \u2192 running headed");
            if (os.equals("linux")) {
                if (!LocalRunExecutor.hasDisplay(env)) {
                    LocalRunExecutor.log(listener, "Linux headed mode with no DISPLAY \u2192 running without xvfb");
                } else if (LocalRunExecutor.hasXvfb(launcher)) {
                    cmd.add("xvfb-run");
                    cmd.add("-a");
                    LocalRunExecutor.log(listener, "DISPLAY set and xvfb present \u2192 using xvfb-run for headed execution");
                } else {
                    throw new IOException("Cannot run headed mode on Linux with DISPLAY set but xvfb is not available");
                }
            }
        }
        cmd.add(binary.getRemote());
        cmd.add("run");
        cmd.add("-k");
        cmd.add(apiKey);
        cmd.add("-u");
        cmd.add(LocalRunExecutor.resolveApiUrl(environment));
        cmd.add("-p");
        cmd.add(projectId.toString());
        cmd.add("-d");
        cmd.add(profileId.toString());
        cmd.add("-b");
        cmd.add(browser);
        if (addHeadlessFlag) {
            cmd.add("-q");
        }
        if (testId != null) {
            cmd.add("-t");
            cmd.add(testId.toString());
        } else {
            cmd.add("-s");
            cmd.add(suiteId.toString());
        }
        cmd.add("-e");
        cmd.add("default.env");
        LocalRunExecutor.log(listener, "Executing sedstart CLI");
        Launcher.ProcStarter ps = launcher.launch();
        ps.cmds(cmd);
        ps.envs((Map)env);
        ps.pwd(sedstartDir);
        ps.stdout((OutputStream)listener.getLogger());
        ps.stderr((OutputStream)listener.getLogger());
        int exit = ps.join();
        LocalRunExecutor.log(listener, "sedstart finished with exit code " + exit);
        if (exit != 0) {
            throw new IOException("sedstart CLI exited with code " + exit);
        }
    }

    private static void downloadBinary(FilePath target, Launcher launcher, String environment, String os, String arch, TaskListener listener) throws IOException, InterruptedException {
        String platform = LocalRunExecutor.resolvePlatform(os, arch);
        String baseUrl = LocalRunExecutor.resolveCliBaseUrl(environment);
        String url = baseUrl + "/" + platform + "/sedstart";
        LocalRunExecutor.log(listener, "Downloading sedstart from: " + url);
        ArrayList<Object> cmd = new ArrayList<Object>();
        if (os.equals("windows")) {
            cmd.add("powershell");
            cmd.add("-Command");
            cmd.add("Invoke-WebRequest -Uri '" + url + "' -OutFile '" + target.getRemote() + "'");
        } else {
            cmd.add("curl");
            cmd.add("-fL");
            cmd.add(url);
            cmd.add("-o");
            cmd.add(target.getRemote());
        }
        int exit = launcher.launch().cmds(cmd).join();
        if (exit != 0) {
            throw new IOException("Failed to download sedstart CLI");
        }
        LocalRunExecutor.log(listener, "Download completed");
    }

    private static void chmodIfNeeded(FilePath binary, Launcher launcher, String os, TaskListener listener) throws IOException, InterruptedException {
        if (os.equals("windows")) {
            return;
        }
        LocalRunExecutor.log(listener, "Making sedstart executable");
        launcher.launch().cmds(new String[]{"chmod", "+x", binary.getRemote()}).join();
    }

    private static String detectOs(Launcher launcher, TaskListener listener) throws IOException, InterruptedException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        launcher.launch().cmds(new String[]{"uname", "-s"}).stdout((OutputStream)out).join();
        String os = out.toString(StandardCharsets.UTF_8).trim().toLowerCase();
        if (os.contains("darwin")) {
            return "darwin";
        }
        if (os.contains("linux")) {
            return "linux";
        }
        return "windows";
    }

    private static String detectArch(Launcher launcher, TaskListener listener) throws IOException, InterruptedException {
        LocalRunExecutor.log(listener, "Detecting architecture");
        if (!launcher.isUnix()) {
            return "amd64";
        }
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        launcher.launch().cmds(new String[]{"uname", "-m"}).stdout((OutputStream)out).join();
        String arch = out.toString(StandardCharsets.UTF_8).trim().toLowerCase();
        if (arch.contains("arm") || arch.contains("aarch64")) {
            return "arm64";
        }
        if (arch.contains("64")) {
            return "amd64";
        }
        if (arch.contains("86")) {
            return "386";
        }
        throw new IOException("Unsupported ARCH: " + arch);
    }

    private static String resolveCliBaseUrl(String env) {
        return "QA".equalsIgnoreCase(env) ? "http://cli.sedinqa.com/latest" : "http://cli.sedstart.com/latest";
    }

    private static String resolveApiUrl(String env) {
        return "QA".equalsIgnoreCase(env) ? "https://sedstart.sedinqa.com/api" : "https://app.sedstart.com/api";
    }

    private static boolean hasDisplay(EnvVars env) {
        String display = (String)env.get((Object)"DISPLAY");
        return display != null && !display.isBlank();
    }

    private static boolean hasXvfb(Launcher launcher) throws IOException, InterruptedException {
        int exit = launcher.launch().cmds(new String[]{"which", "xvfb-run"}).join();
        return exit == 0;
    }

    private static String resolvePlatform(String os, String arch) throws IOException {
        if (os.contains("darwin") || os.contains("mac")) {
            return arch.contains("arm") ? "cli_darwin_arm64_v8.0" : "cli_darwin_amd64_v1";
        }
        if (os.contains("linux")) {
            return arch.contains("arm") ? "cli_linux_arm64_v8.0" : "cli_linux_amd64_v1";
        }
        if (os.contains("win")) {
            return arch.contains("arm") ? "cli_windows_arm64_v8.0" : "cli_windows_amd64_v1";
        }
        throw new IOException("Unsupported OS/arch: " + os + " / " + arch);
    }

    private static void validate(Integer projectId, Integer suiteId, Integer testId, Integer profileId) throws IOException {
        if (projectId == null) {
            throw new IOException("projectId is required");
        }
        if (profileId == null) {
            throw new IOException("profileId is required");
        }
        if (suiteId == null && testId == null || suiteId != null && testId != null) {
            throw new IOException("Provide exactly one of suiteId or testId");
        }
    }

    private static void log(TaskListener listener, String msg) {
        listener.getLogger().println("[sedstart] " + msg);
    }
}

