package io.jenkins.plugins.formatter;

import com.google.common.base.Throwables;
import edu.umd.cs.findbugs.annotations.NonNull;
import hudson.Extension;
import hudson.markup.MarkupFormatter;
import hudson.markup.MarkupFormatterDescriptor;
import jenkins.model.Jenkins;
import org.kohsuke.stapler.DataBoundConstructor;
import org.owasp.html.Handler;
import org.owasp.html.HtmlSanitizer;
import org.owasp.html.HtmlStreamRenderer;
import org.owasp.html.PolicyFactory;

import java.io.*;
import java.lang.reflect.InvocationTargetException;

public class CustomMarkupFormatter extends MarkupFormatter {

    final boolean disableSyntaxHighlighting;

    @DataBoundConstructor
    public CustomMarkupFormatter(final boolean disableSyntaxHighlighting) {
        this.disableSyntaxHighlighting = disableSyntaxHighlighting;
    }

    @Override
    public void translate(String s, @NonNull Writer writer) throws IOException {

        HtmlStreamRenderer renderer = HtmlStreamRenderer.create(
                writer,
                // Receives notifications on a failure to write to the output.
                Handler.PROPAGATE, // System.out suppresses IOExceptions
                // Our HTML parser is very lenient, but this receives notifications on
                // truly bizarre inputs.
                x -> {
                    throw new Error(x);
                }
        );

        PolicyFactory DEFINITION = null;
        String policyDefinition = null;

        // Safely retrieve the policy configuration
        PolicyConfiguration config = PolicyConfiguration.get();

        // If get() returns null, try accessing via extension list
        if (config == null) {
            try {
                java.util.List<PolicyConfiguration> configs = Jenkins.get().getExtensionList(PolicyConfiguration.class);
                if (!configs.isEmpty()) {
                    config = configs.get(0);
                }
            } catch (Exception e) {
                // Ignore and fall back to default policy
            }
        }

        if (config != null) {
            policyDefinition = config.getPolicyDefinition();
        }

        // Try to build policy from configuration
        if (policyDefinition != null && !policyDefinition.isEmpty()) {
            try {
                DEFINITION = CustomPolicyBuilder.build(policyDefinition);
                HtmlSanitizer.sanitize(s, DEFINITION.apply(renderer));
            } catch (IllegalAccessException | NoSuchMethodException | InvocationTargetException | DefinedException e) {
                e.printStackTrace();
                // Fall through to use default policy
                DEFINITION = null;
            }
        }

        // Fall back to default policy if custom policy failed or wasn't available
        if (DEFINITION == null) {
            try {
                DEFINITION = CustomPolicyBuilder.build(PolicyConfiguration.DEFAULT_POLICY);
                HtmlSanitizer.sanitize(s, DEFINITION.apply(renderer));
            } catch (IllegalAccessException | NoSuchMethodException | InvocationTargetException | DefinedException e) {
                e.printStackTrace();
            }
        }

    }

    @Extension
    public static class DescriptorImpl extends MarkupFormatterDescriptor {
        @Override
        public String getDisplayName() {
            return "Customizable HTML Formatter";
        }
    }

    public static final MarkupFormatter INSTANCE = new CustomMarkupFormatter(false);
}
