package edu.hm.hafner.coverage.parser;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.StartElement;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import edu.hm.hafner.coverage.ClassNode;
import edu.hm.hafner.coverage.Coverage;
import edu.hm.hafner.coverage.Coverage.CoverageBuilder;
import edu.hm.hafner.coverage.CoverageParser;
import edu.hm.hafner.coverage.FileNode;
import edu.hm.hafner.coverage.MethodNode;
import edu.hm.hafner.coverage.Metric;
import edu.hm.hafner.coverage.ModuleNode;
import edu.hm.hafner.coverage.Node;
import edu.hm.hafner.coverage.PackageNode;
import edu.hm.hafner.coverage.Value;
import edu.hm.hafner.util.FilteredLog;
import edu.hm.hafner.util.PathUtil;
import edu.hm.hafner.util.SecureXmlParserFactory;

import java.io.Reader;
import java.io.Serial;
import java.nio.file.Path;
import java.util.UUID;
import java.util.regex.Pattern;

/**
 * Parses Cobertura reports into a hierarchical Java Object Model.
 *
 * @author Melissa Bauer
 * @author Ullrich Hafner
 */
@SuppressWarnings({"checkstyle:ClassDataAbstractionCoupling", "PMD.GodClass", "PMD.AvoidDeeplyNestedIfStmts"})
public class CoberturaParser extends CoverageParser {
    @Serial
    private static final long serialVersionUID = -3625341318291829577L;

    private static final Pattern BRANCH_PATTERN = Pattern.compile(".*\\((?<covered>\\d+)/(?<total>\\d+)\\)");
    private static final PathUtil PATH_UTIL = new PathUtil();

    private static final String DETERMINISTIC_PATH_PREFIX = "/_/";

    private static final Coverage DEFAULT_BRANCH_COVERAGE = new CoverageBuilder(Metric.BRANCH).withCovered(2).withMissed(0).build();
    private static final Coverage LINE_COVERED = new CoverageBuilder(Metric.LINE).withCovered(1).withMissed(0).build();
    private static final Coverage LINE_MISSED = new CoverageBuilder(Metric.LINE).withCovered(0).withMissed(1).build();

    /** XML elements. */
    private static final QName SOURCE = new QName("source");
    private static final QName PACKAGE = new QName("package");
    private static final QName CLASS = new QName("class");
    private static final QName METHOD = new QName("method");
    private static final QName LINE = new QName("line");

    /** Required attributes of the XML elements. */
    private static final QName NAME = new QName("name");
    private static final QName FILE_NAME = new QName("filename");
    private static final QName SIGNATURE = new QName("signature");
    private static final QName HITS = new QName("hits");
    private static final QName COMPLEXITY = new QName("complexity");
    private static final QName NUMBER = new QName("number");

    /** Optional attributes of the XML elements. */
    private static final QName BRANCH = new QName("branch");
    private static final QName CONDITION_COVERAGE = new QName("condition-coverage");

    /**
     * Creates a new instance of {@link CoberturaParser}.
     */
    public CoberturaParser() {
        this(ProcessingMode.FAIL_FAST);
    }

    /**
     * Creates a new instance of {@link CoberturaParser}.
     *
     * @param processingMode
     *         determines whether to ignore errors
     */
    public CoberturaParser(final ProcessingMode processingMode) {
        super(processingMode);
    }

    @Override
    protected ModuleNode parseReport(final Reader reader, final String fileName, final FilteredLog log) {
        try {
            var eventReader = new SecureXmlParserFactory().createXmlEventReader(reader);

            var root = new ModuleNode(EMPTY); // Cobertura has no support for module names
            handleEmptyResults(fileName, log, readModule(eventReader, root, fileName, log));
            return root;
        }
        catch (XMLStreamException exception) {
            throw new ParsingException(exception);
        }
    }

    private boolean readModule(final XMLEventReader eventReader, final ModuleNode root,
            final String fileName, final FilteredLog log) throws XMLStreamException {
        boolean isEmpty = true;

        while (eventReader.hasNext()) {
            var event = eventReader.nextEvent();

            if (event.isStartElement()) {
                var startElement = event.asStartElement();
                var tagName = startElement.getName();
                if (SOURCE.equals(tagName)) {
                    readSource(eventReader, root);
                }
                else if (PACKAGE.equals(tagName)) {
                    readPackage(eventReader, root, readName(startElement), fileName, log);
                    isEmpty = false;
                }
            }
        }

        return isEmpty;
    }

    private void readPackage(final XMLEventReader reader, final ModuleNode root,
            final String packageName, final String fileName, final FilteredLog log) throws XMLStreamException {
        var packageNode = root.findOrCreatePackageNode(packageName);

        while (reader.hasNext()) {
            var event = reader.nextEvent();

            if (event.isStartElement()) {
                var element = event.asStartElement();
                if (CLASS.equals(element.getName())) {
                    var fileNode = createFileNode(element, packageNode);

                    readClassOrMethod(reader, fileNode, fileNode, element, fileName, log);
                }
            }
            else if (event.isEndElement()) {
                return; // finish processing of package
            }
        }
    }

    private FileNode createFileNode(final StartElement element, final PackageNode packageNode) {
        var fileName = getValueOf(element, FILE_NAME);
        var relativePath = Strings.CS.removeStart(PATH_UTIL.getRelativePath(fileName), DETERMINISTIC_PATH_PREFIX);
        var path = getTreeStringBuilder().intern(relativePath);

        return packageNode.findOrCreateFileNode(getFileName(fileName), path);
    }

    private String getFileName(final String relativePath) {
        var path = Path.of(PATH_UTIL.getAbsolutePath(relativePath)).getFileName();
        if (path == null) {
            return relativePath;
        }
        return path.toString();
    }

    @SuppressWarnings({"PMD.CyclomaticComplexity", "PMD.CognitiveComplexity"})
    protected void readClassOrMethod(final XMLEventReader reader, final FileNode fileNode,
            final Node parentNode, final StartElement element, final String fileName, final FilteredLog log)
            throws XMLStreamException {
        var node = createNode(parentNode, element, log);
        getOptionalValueOf(element, COMPLEXITY)
                .ifPresent(c -> node.addValue(new Value(Metric.CYCLOMATIC_COMPLEXITY, readComplexity(c))));

        var coveragePerLine = new HashMap<Integer, List<Coverage>>();

        while (reader.hasNext()) {
            var event = reader.nextEvent();

            if (event.isStartElement()) {
                var nextElement = event.asStartElement();
                if (LINE.equals(nextElement.getName())) {
                    processLineElement(nextElement, coveragePerLine);
                }
                else if (METHOD.equals(nextElement.getName())) {
                    readClassOrMethod(reader, fileNode, node, nextElement, fileName, log); // recursive call
                }
            }
            else if (event.isEndElement()) {
                var endElement = event.asEndElement();
                if (CLASS.equals(endElement.getName()) || METHOD.equals(endElement.getName())) {
                    if (CLASS.equals(endElement.getName())) {
                        coveragePerLine.forEach((lineNumber, coverages) ->
                                addFileNodeCounters(fileNode, lineNumber, coverages));
                    }

                    var coverages = recalculateCoverageFromMergedLines(coveragePerLine);
                    node.addValue(coverages[0]);
                    if (coverages[1].isSet()) {
                        node.addValue(coverages[1]);
                    }
                    return;
                }
            }
        }
        throw createEofException(fileName);
    }

    private void addFileNodeCounters(final FileNode fileNode, final Integer lineNumber,
            final List<Coverage> coverages) {
        var hasBranchCoverage = coverages.stream()
                .anyMatch(c -> c.getMetric() == Metric.BRANCH);

        if (hasBranchCoverage) {
            var branchCoverage = coverages.stream()
                    .filter(c -> c.getMetric() == Metric.BRANCH)
                    .reduce(Coverage.nullObject(Metric.BRANCH), Coverage::add);
            var covered = branchCoverage.getCovered();
            var missed = branchCoverage.getMissed();
            fileNode.addCounters(lineNumber, covered, missed);
        }
        else {
            var lineCoverage = coverages.stream()
                    .filter(c -> c.getMetric() == Metric.LINE)
                    .reduce(Coverage.nullObject(Metric.LINE), Coverage::add);
            var covered = lineCoverage.getCovered();
            var missed = lineCoverage.getMissed();
            fileNode.addCounters(lineNumber, covered, missed);
        }
    }

    /**
     * Merges duplicate line coverage entries.
     * <ul>
     * <li>For line coverage (no branches): keeps the line as covered if any entry has hits > 0</li>
     * <li>For branch coverage: keeps the maximum covered branches</li>
     * </ul>
     *
     * @param existing the existing list of coverages for the line
     * @param newCoverages the new list of coverages to merge
     * @return the merged list of coverages
     */
    private List<Coverage> mergeDuplicateLines(final List<Coverage> existing, final List<Coverage> newCoverages) {
        var result = new ArrayList<Coverage>();

        var existingLine = existing.stream()
                .filter(c -> c.getMetric() == Metric.LINE)
                .findFirst();
        var newLine = newCoverages.stream()
                .filter(c -> c.getMetric() == Metric.LINE)
                .findFirst();

        if (existingLine.isPresent() && newLine.isPresent()) {
            result.add(mergeLineCoverage(existingLine.get(), newLine.get()));
        }
        else {
            existingLine.ifPresent(result::add);
            newLine.ifPresent(result::add);
        }

        var existingBranch = existing.stream()
                .filter(c -> c.getMetric() == Metric.BRANCH)
                .findFirst();
        var newBranch = newCoverages.stream()
                .filter(c -> c.getMetric() == Metric.BRANCH)
                .findFirst();

        if (existingBranch.isPresent() && newBranch.isPresent()) {
            result.add(mergeBranchCoverage(existingBranch.get(), newBranch.get()));
        }
        else {
            existingBranch.ifPresent(result::add);
            newBranch.ifPresent(result::add);
        }

        return result;
    }

    private Coverage mergeLineCoverage(final Coverage existing, final Coverage newCoverage) {
        return hasAnyCovered(existing, newCoverage) ? LINE_COVERED : LINE_MISSED;
    }

    private boolean hasAnyCovered(final Coverage existing, final Coverage newCoverage) {
        return existing.getCovered() > 0 || newCoverage.getCovered() > 0;
    }

    private Coverage mergeBranchCoverage(final Coverage existing, final Coverage newCoverage) {
        return newCoverage.getCovered() >= existing.getCovered() ? newCoverage : existing;
    }

    private void processLineElement(final StartElement nextElement,
            final Map<Integer, List<Coverage>> coveragePerLine) {
        int lineNumber = getIntegerValueOf(nextElement, NUMBER);
        int lineHits = getIntegerValueOf(nextElement, HITS);

        var coverages = new ArrayList<Coverage>();

        coverages.add(computeLineCoverage(lineHits));

        if (isBranchCoverage(nextElement)) {
            coverages.add(readBranchCoverage(nextElement));
        }

        coveragePerLine.merge(lineNumber, coverages, this::mergeDuplicateLines);
    }

    private Coverage[] recalculateCoverageFromMergedLines(final Map<Integer, List<Coverage>> coveragePerLine) {
        var lineCoverage = coveragePerLine.values().stream()
                .flatMap(List::stream)
                .filter(c -> c.getMetric() == Metric.LINE)
                .reduce(Coverage.nullObject(Metric.LINE), Coverage::add);

        var branchCoverage = coveragePerLine.values().stream()
                .flatMap(List::stream)
                .filter(c -> c.getMetric() == Metric.BRANCH)
                .reduce(Coverage.nullObject(Metric.BRANCH), Coverage::add);

        return new Coverage[] {lineCoverage, branchCoverage};
    }

    protected Coverage computeLineCoverage(final int coverage) {
        return coverage > 0 ? LINE_COVERED : LINE_MISSED;
    }

    protected Node createNode(final Node parentNode, final StartElement element, final FilteredLog log) {
        var name = readName(element);
        if (CLASS.equals(element.getName())) {
            return createClassNode(parentNode, log, name);
        }

        return createMethodNode(parentNode, element, log, name);
    }

    private MethodNode createMethodNode(final Node parentNode, final StartElement element, final FilteredLog log,
            final String name) {
        var methodName = name;
        var signature = getValueOf(element, SIGNATURE);
        if (parentNode.findMethod(methodName, signature).isPresent() && ignoreErrors()) {
            log.logError("Found a duplicate method '%s' with signature '%s' in '%s'",
                    methodName, signature, parentNode.getName());
            methodName = name + "-" + createId();
        }
        return parentNode.createMethodNode(methodName, signature);
    }

    private ClassNode createClassNode(final Node parentNode, final FilteredLog log, final String name) {
        var className = name;
        if (parentNode.hasChild(className) && ignoreErrors()) {
            log.logError("Found a duplicate class '%s' in '%s'", className, parentNode.getName());
            className = name + "-" + createId();
        }
        return parentNode.createClassNode(className);
    }

    private String readName(final StartElement element) {
        return StringUtils.defaultIfBlank(getValueOf(element, NAME), createId());
    }

    private String createId() {
        return UUID.randomUUID().toString();
    }

    protected int readComplexity(final String c) {
        try {
            return Math.round(Float.parseFloat(c)); // some reports use float values
        }
        catch (NumberFormatException ignore) {
            return 0;
        }
    }

    protected boolean isBranchCoverage(final StartElement line) {
        return getOptionalValueOf(line, BRANCH)
                .map(Boolean::parseBoolean)
                .orElse(false);
    }

    private void readSource(final XMLEventReader reader, final ModuleNode root) throws XMLStreamException {
        var aggregatedContent = new StringBuilder();

        while (reader.hasNext()) {
            var event = reader.nextEvent();
            if (event.isCharacters()) {
                aggregatedContent.append(event.asCharacters().getData());
            }
            else if (event.isEndElement()) {
                root.addSource(new PathUtil().getRelativePath(aggregatedContent.toString()));

                return;
            }
        }
    }

    protected Coverage readBranchCoverage(final StartElement line) {
        return getOptionalValueOf(line, CONDITION_COVERAGE).map(this::fromConditionCoverage).orElse(DEFAULT_BRANCH_COVERAGE);
    }

    private Coverage fromConditionCoverage(final String conditionCoverageAttribute) {
        var matcher = BRANCH_PATTERN.matcher(conditionCoverageAttribute);
        if (matcher.matches()) {
            return new CoverageBuilder().withMetric(Metric.BRANCH)
                    .withCovered(matcher.group("covered"))
                    .withTotal(matcher.group("total"))
                    .build();
        }
        return Coverage.nullObject(Metric.BRANCH);
    }
}
